/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.J2ktIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.DoubleUtils;
import com.google.common.math.ElementTypesAreNonnullByDefault;
import com.google.common.math.IntMath;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.Longs;
import com.google.common.primitives.UnsignedLongs;
import java.math.RoundingMode;

@ElementTypesAreNonnullByDefault
@GwtCompatible(emulated=true)
public final class LongMath {
    @VisibleForTesting
    static final long MAX_SIGNED_POWER_OF_TWO = 0x4000000000000000L;
    @VisibleForTesting
    static final long MAX_POWER_OF_SQRT2_UNSIGNED = -5402926248376769404L;
    @VisibleForTesting
    static final byte[] maxLog10ForLeadingZeros = new byte[]{19, 18, 18, 18, 18, 17, 17, 17, 16, 16, 16, 15, 15, 15, 15, 14, 14, 14, 13, 13, 13, 12, 12, 12, 12, 11, 11, 11, 10, 10, 10, 9, 9, 9, 9, 8, 8, 8, 7, 7, 7, 6, 6, 6, 6, 5, 5, 5, 4, 4, 4, 3, 3, 3, 3, 2, 2, 2, 1, 1, 1, 0, 0, 0};
    @J2ktIncompatible
    @GwtIncompatible
    @VisibleForTesting
    static final long[] powersOf10 = new long[]{1L, 10L, 100L, 1000L, 10000L, 100000L, 1000000L, 10000000L, 100000000L, 1000000000L, 10000000000L, 100000000000L, 1000000000000L, 10000000000000L, 100000000000000L, 1000000000000000L, 10000000000000000L, 100000000000000000L, 1000000000000000000L};
    @J2ktIncompatible
    @GwtIncompatible
    @VisibleForTesting
    static final long[] halfPowersOf10 = new long[]{3L, 31L, 316L, 3162L, 31622L, 316227L, 3162277L, 31622776L, 316227766L, 3162277660L, 31622776601L, 316227766016L, 3162277660168L, 31622776601683L, 316227766016837L, 3162277660168379L, 31622776601683793L, 316227766016837933L, 3162277660168379331L};
    @VisibleForTesting
    static final long FLOOR_SQRT_MAX_LONG = 3037000499L;
    static final long[] factorials = new long[]{1L, 1L, 2L, 6L, 24L, 120L, 720L, 5040L, 40320L, 362880L, 3628800L, 39916800L, 479001600L, 6227020800L, 87178291200L, 1307674368000L, 20922789888000L, 355687428096000L, 6402373705728000L, 121645100408832000L, 2432902008176640000L};
    static final int[] biggestBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, 3810779, 121977, 16175, 4337, 1733, 887, 534, 361, 265, 206, 169, 143, 125, 111, 101, 94, 88, 83, 79, 76, 74, 72, 70, 69, 68, 67, 67, 66, 66, 66, 66};
    @VisibleForTesting
    static final int[] biggestSimpleBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE, 2642246, 86251, 11724, 3218, 1313, 684, 419, 287, 214, 169, 139, 119, 105, 95, 87, 81, 76, 73, 70, 68, 66, 64, 63, 62, 62, 61, 61, 61};
    private static final int SIEVE_30 = -545925251;
    private static final long[][] millerRabinBaseSets = new long[][]{{291830L, 126401071349994536L}, {885594168L, 725270293939359937L, 3569819667048198375L}, {273919523040L, 15L, 7363882082L, 992620450144556L}, {47636622961200L, 2L, 2570940L, 211991001L, 3749873356L}, {7999252175582850L, 2L, 4130806001517L, 149795463772692060L, 186635894390467037L, 3967304179347715805L}, {585226005592931976L, 2L, 123635709730000L, 9233062284813009L, 43835965440333360L, 761179012939631437L, 1263739024124850375L}, {Long.MAX_VALUE, 2L, 325L, 9375L, 28178L, 450775L, 9780504L, 1795265022L}};

    public static long ceilingPowerOfTwo(long x) {
        MathPreconditions.checkPositive("x", x);
        if (x > 0x4000000000000000L) {
            throw new ArithmeticException("ceilingPowerOfTwo(" + x + ") is not representable as a long");
        }
        return 1L << -Long.numberOfLeadingZeros(x - 1L);
    }

    public static long floorPowerOfTwo(long x) {
        MathPreconditions.checkPositive("x", x);
        return 1L << 63 - Long.numberOfLeadingZeros(x);
    }

    public static boolean isPowerOfTwo(long x) {
        return x > 0L & (x & x - 1L) == 0L;
    }

    @VisibleForTesting
    static int lessThanBranchFree(long x, long y) {
        return (int)((x - y ^ 0xFFFFFFFFFFFFFFFFL ^ 0xFFFFFFFFFFFFFFFFL) >>> 63);
    }

    public static int log2(long x, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(LongMath.isPowerOfTwo(x));
            }
            case DOWN: 
            case FLOOR: {
                return 63 - Long.numberOfLeadingZeros(x);
            }
            case UP: 
            case CEILING: {
                return 64 - Long.numberOfLeadingZeros(x - 1L);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int leadingZeros = Long.numberOfLeadingZeros(x);
                long cmp = -5402926248376769404L >>> leadingZeros;
                int logFloor = 63 - leadingZeros;
                return logFloor + LongMath.lessThanBranchFree(cmp, x);
            }
        }
        throw new AssertionError((Object)"impossible");
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static int log10(long x, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x);
        int logFloor = LongMath.log10Floor(x);
        long floorPow = powersOf10[logFloor];
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(x == floorPow);
            }
            case DOWN: 
            case FLOOR: {
                return logFloor;
            }
            case UP: 
            case CEILING: {
                return logFloor + LongMath.lessThanBranchFree(floorPow, x);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                return logFloor + LongMath.lessThanBranchFree(halfPowersOf10[logFloor], x);
            }
        }
        throw new AssertionError();
    }

    @J2ktIncompatible
    @GwtIncompatible
    static int log10Floor(long x) {
        byte y = maxLog10ForLeadingZeros[Long.numberOfLeadingZeros(x)];
        return y - LongMath.lessThanBranchFree(x, powersOf10[y]);
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static long pow(long b2, int k2) {
        MathPreconditions.checkNonNegative("exponent", k2);
        if (-2L <= b2 && b2 <= 2L) {
            switch ((int)b2) {
                case 0: {
                    return k2 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (k2 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    return k2 < 64 ? 1L << k2 : 0L;
                }
                case -2: {
                    if (k2 < 64) {
                        return (k2 & 1) == 0 ? 1L << k2 : -(1L << k2);
                    }
                    return 0L;
                }
            }
            throw new AssertionError();
        }
        long accum = 1L;
        while (true) {
            switch (k2) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return accum * b2;
                }
            }
            accum *= (k2 & 1) == 0 ? 1L : b2;
            b2 *= b2;
            k2 >>= 1;
        }
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static long sqrt(long x, RoundingMode mode) {
        MathPreconditions.checkNonNegative("x", x);
        if (LongMath.fitsInInt(x)) {
            return IntMath.sqrt((int)x, mode);
        }
        long guess = (long)Math.sqrt(x);
        long guessSquared = guess * guess;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(guessSquared == x);
                return guess;
            }
            case DOWN: 
            case FLOOR: {
                if (x < guessSquared) {
                    return guess - 1L;
                }
                return guess;
            }
            case UP: 
            case CEILING: {
                if (x > guessSquared) {
                    return guess + 1L;
                }
                return guess;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                long sqrtFloor = guess - (long)(x < guessSquared ? 1 : 0);
                long halfSquare = sqrtFloor * sqrtFloor + sqrtFloor;
                return sqrtFloor + (long)LongMath.lessThanBranchFree(halfSquare, x);
            }
        }
        throw new AssertionError();
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static long divide(long p, long q, RoundingMode mode) {
        boolean increment;
        Preconditions.checkNotNull(mode);
        long div = p / q;
        long rem = p - q * div;
        if (rem == 0L) {
            return div;
        }
        int signum = 1 | (int)((p ^ q) >> 63);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(rem == 0L);
            }
            case DOWN: {
                increment = false;
                break;
            }
            case UP: {
                increment = true;
                break;
            }
            case CEILING: {
                increment = signum > 0;
                break;
            }
            case FLOOR: {
                increment = signum < 0;
                break;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                long absRem = Math.abs(rem);
                long cmpRemToHalfDivisor = absRem - (Math.abs(q) - absRem);
                if (cmpRemToHalfDivisor == 0L) {
                    increment = mode == RoundingMode.HALF_UP || mode == RoundingMode.HALF_EVEN && (div & 1L) != 0L;
                    break;
                }
                increment = cmpRemToHalfDivisor > 0L;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return increment ? div + (long)signum : div;
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static int mod(long x, int m4) {
        return (int)LongMath.mod(x, (long)m4);
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static long mod(long x, long m4) {
        if (m4 <= 0L) {
            throw new ArithmeticException("Modulus must be positive");
        }
        long result2 = x % m4;
        return result2 >= 0L ? result2 : result2 + m4;
    }

    public static long gcd(long a2, long b2) {
        MathPreconditions.checkNonNegative("a", a2);
        MathPreconditions.checkNonNegative("b", b2);
        if (a2 == 0L) {
            return b2;
        }
        if (b2 == 0L) {
            return a2;
        }
        int aTwos = Long.numberOfTrailingZeros(a2);
        a2 >>= aTwos;
        int bTwos = Long.numberOfTrailingZeros(b2);
        b2 >>= bTwos;
        while (a2 != b2) {
            long delta = a2 - b2;
            long minDeltaOrZero = delta & delta >> 63;
            a2 = delta - minDeltaOrZero - minDeltaOrZero;
            b2 += minDeltaOrZero;
            a2 >>= Long.numberOfTrailingZeros(a2);
        }
        return a2 << Math.min(aTwos, bTwos);
    }

    public static long checkedAdd(long a2, long b2) {
        long result2 = a2 + b2;
        MathPreconditions.checkNoOverflow((a2 ^ b2) < 0L | (a2 ^ result2) >= 0L, "checkedAdd", a2, b2);
        return result2;
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static long checkedSubtract(long a2, long b2) {
        long result2 = a2 - b2;
        MathPreconditions.checkNoOverflow((a2 ^ b2) >= 0L | (a2 ^ result2) >= 0L, "checkedSubtract", a2, b2);
        return result2;
    }

    public static long checkedMultiply(long a2, long b2) {
        int leadingZeros = Long.numberOfLeadingZeros(a2) + Long.numberOfLeadingZeros(a2 ^ 0xFFFFFFFFFFFFFFFFL) + Long.numberOfLeadingZeros(b2) + Long.numberOfLeadingZeros(b2 ^ 0xFFFFFFFFFFFFFFFFL);
        if (leadingZeros > 65) {
            return a2 * b2;
        }
        MathPreconditions.checkNoOverflow(leadingZeros >= 64, "checkedMultiply", a2, b2);
        MathPreconditions.checkNoOverflow(a2 >= 0L | b2 != Long.MIN_VALUE, "checkedMultiply", a2, b2);
        long result2 = a2 * b2;
        MathPreconditions.checkNoOverflow(a2 == 0L || result2 / a2 == b2, "checkedMultiply", a2, b2);
        return result2;
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static long checkedPow(long b2, int k2) {
        MathPreconditions.checkNonNegative("exponent", k2);
        if (b2 >= -2L & b2 <= 2L) {
            switch ((int)b2) {
                case 0: {
                    return k2 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (k2 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    MathPreconditions.checkNoOverflow(k2 < 63, "checkedPow", b2, (long)k2);
                    return 1L << k2;
                }
                case -2: {
                    MathPreconditions.checkNoOverflow(k2 < 64, "checkedPow", b2, (long)k2);
                    return (k2 & 1) == 0 ? 1L << k2 : -1L << k2;
                }
            }
            throw new AssertionError();
        }
        long accum = 1L;
        while (true) {
            switch (k2) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return LongMath.checkedMultiply(accum, b2);
                }
            }
            if ((k2 & 1) != 0) {
                accum = LongMath.checkedMultiply(accum, b2);
            }
            if ((k2 >>= 1) <= 0) continue;
            MathPreconditions.checkNoOverflow(-3037000499L <= b2 && b2 <= 3037000499L, "checkedPow", b2, (long)k2);
            b2 *= b2;
        }
    }

    public static long saturatedAdd(long a2, long b2) {
        long naiveSum;
        if ((a2 ^ b2) < 0L | (a2 ^ (naiveSum = a2 + b2)) >= 0L) {
            return naiveSum;
        }
        return Long.MAX_VALUE + (naiveSum >>> 63 ^ 1L);
    }

    public static long saturatedSubtract(long a2, long b2) {
        long naiveDifference;
        if ((a2 ^ b2) >= 0L | (a2 ^ (naiveDifference = a2 - b2)) >= 0L) {
            return naiveDifference;
        }
        return Long.MAX_VALUE + (naiveDifference >>> 63 ^ 1L);
    }

    public static long saturatedMultiply(long a2, long b2) {
        int leadingZeros = Long.numberOfLeadingZeros(a2) + Long.numberOfLeadingZeros(a2 ^ 0xFFFFFFFFFFFFFFFFL) + Long.numberOfLeadingZeros(b2) + Long.numberOfLeadingZeros(b2 ^ 0xFFFFFFFFFFFFFFFFL);
        if (leadingZeros > 65) {
            return a2 * b2;
        }
        long limit = Long.MAX_VALUE + ((a2 ^ b2) >>> 63);
        if (leadingZeros < 64 | a2 < 0L & b2 == Long.MIN_VALUE) {
            return limit;
        }
        long result2 = a2 * b2;
        if (a2 == 0L || result2 / a2 == b2) {
            return result2;
        }
        return limit;
    }

    public static long saturatedPow(long b2, int k2) {
        MathPreconditions.checkNonNegative("exponent", k2);
        if (b2 >= -2L & b2 <= 2L) {
            switch ((int)b2) {
                case 0: {
                    return k2 == 0 ? 1L : 0L;
                }
                case 1: {
                    return 1L;
                }
                case -1: {
                    return (k2 & 1) == 0 ? 1L : -1L;
                }
                case 2: {
                    if (k2 >= 63) {
                        return Long.MAX_VALUE;
                    }
                    return 1L << k2;
                }
                case -2: {
                    if (k2 >= 64) {
                        return Long.MAX_VALUE + (long)(k2 & 1);
                    }
                    return (k2 & 1) == 0 ? 1L << k2 : -1L << k2;
                }
            }
            throw new AssertionError();
        }
        long accum = 1L;
        long limit = Long.MAX_VALUE + (b2 >>> 63 & (long)(k2 & 1));
        while (true) {
            switch (k2) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return LongMath.saturatedMultiply(accum, b2);
                }
            }
            if ((k2 & 1) != 0) {
                accum = LongMath.saturatedMultiply(accum, b2);
            }
            if ((k2 >>= 1) <= 0) continue;
            if (-3037000499L > b2 | b2 > 3037000499L) {
                return limit;
            }
            b2 *= b2;
        }
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static long factorial(int n) {
        MathPreconditions.checkNonNegative("n", n);
        return n < factorials.length ? factorials[n] : Long.MAX_VALUE;
    }

    public static long binomial(int n, int k2) {
        MathPreconditions.checkNonNegative("n", n);
        MathPreconditions.checkNonNegative("k", k2);
        Preconditions.checkArgument(k2 <= n, "k (%s) > n (%s)", k2, n);
        if (k2 > n >> 1) {
            k2 = n - k2;
        }
        switch (k2) {
            case 0: {
                return 1L;
            }
            case 1: {
                return n;
            }
        }
        if (n < factorials.length) {
            return factorials[n] / (factorials[k2] * factorials[n - k2]);
        }
        if (k2 >= biggestBinomials.length || n > biggestBinomials[k2]) {
            return Long.MAX_VALUE;
        }
        if (k2 < biggestSimpleBinomials.length && n <= biggestSimpleBinomials[k2]) {
            long result2 = n--;
            for (int i2 = 2; i2 <= k2; ++i2) {
                result2 *= (long)n;
                result2 /= (long)i2;
                --n;
            }
            return result2;
        }
        int nBits = LongMath.log2(n, RoundingMode.CEILING);
        long result3 = 1L;
        long numerator = n--;
        long denominator = 1L;
        int numeratorBits = nBits;
        int i3 = 2;
        while (i3 <= k2) {
            if (numeratorBits + nBits < 63) {
                numerator *= (long)n;
                denominator *= (long)i3;
                numeratorBits += nBits;
            } else {
                result3 = LongMath.multiplyFraction(result3, numerator, denominator);
                numerator = n;
                denominator = i3;
                numeratorBits = nBits;
            }
            ++i3;
            --n;
        }
        return LongMath.multiplyFraction(result3, numerator, denominator);
    }

    static long multiplyFraction(long x, long numerator, long denominator) {
        if (x == 1L) {
            return numerator / denominator;
        }
        long commonDivisor = LongMath.gcd(x, denominator);
        return (x /= commonDivisor) * (numerator / (denominator /= commonDivisor));
    }

    static boolean fitsInInt(long x) {
        return (long)((int)x) == x;
    }

    public static long mean(long x, long y) {
        return (x & y) + ((x ^ y) >> 1);
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static boolean isPrime(long n) {
        if (n < 2L) {
            MathPreconditions.checkNonNegative("n", n);
            return false;
        }
        if (n < 66L) {
            long mask = 722865708377213483L;
            return (mask >> (int)n - 2 & 1L) != 0L;
        }
        if ((0xDF75D77D & 1 << (int)(n % 30L)) != 0) {
            return false;
        }
        if (n % 7L == 0L || n % 11L == 0L || n % 13L == 0L) {
            return false;
        }
        if (n < 289L) {
            return true;
        }
        for (long[] baseSet : millerRabinBaseSets) {
            if (n > baseSet[0]) continue;
            for (int i2 = 1; i2 < baseSet.length; ++i2) {
                if (MillerRabinTester.test(baseSet[i2], n)) continue;
                return false;
            }
            return true;
        }
        throw new AssertionError();
    }

    @J2ktIncompatible
    @GwtIncompatible
    public static double roundToDouble(long x, RoundingMode mode) {
        double roundArbitrarily = x;
        long roundArbitrarilyAsLong = (long)roundArbitrarily;
        int cmpXToRoundArbitrarily = roundArbitrarilyAsLong == Long.MAX_VALUE ? -1 : Longs.compare(x, roundArbitrarilyAsLong);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(cmpXToRoundArbitrarily == 0);
                return roundArbitrarily;
            }
            case FLOOR: {
                return cmpXToRoundArbitrarily >= 0 ? roundArbitrarily : DoubleUtils.nextDown(roundArbitrarily);
            }
            case CEILING: {
                return cmpXToRoundArbitrarily <= 0 ? roundArbitrarily : Math.nextUp(roundArbitrarily);
            }
            case DOWN: {
                if (x >= 0L) {
                    return cmpXToRoundArbitrarily >= 0 ? roundArbitrarily : DoubleUtils.nextDown(roundArbitrarily);
                }
                return cmpXToRoundArbitrarily <= 0 ? roundArbitrarily : Math.nextUp(roundArbitrarily);
            }
            case UP: {
                if (x >= 0L) {
                    return cmpXToRoundArbitrarily <= 0 ? roundArbitrarily : Math.nextUp(roundArbitrarily);
                }
                return cmpXToRoundArbitrarily >= 0 ? roundArbitrarily : DoubleUtils.nextDown(roundArbitrarily);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int diff;
                long roundCeiling;
                double roundCeilingAsDouble;
                long roundFloor;
                double roundFloorAsDouble;
                if (cmpXToRoundArbitrarily >= 0) {
                    roundFloorAsDouble = roundArbitrarily;
                    roundFloor = roundArbitrarilyAsLong;
                    roundCeilingAsDouble = Math.nextUp(roundArbitrarily);
                    roundCeiling = (long)Math.ceil(roundCeilingAsDouble);
                } else {
                    roundCeilingAsDouble = roundArbitrarily;
                    roundCeiling = roundArbitrarilyAsLong;
                    roundFloorAsDouble = DoubleUtils.nextDown(roundArbitrarily);
                    roundFloor = (long)Math.floor(roundFloorAsDouble);
                }
                long deltaToFloor = x - roundFloor;
                long deltaToCeiling = roundCeiling - x;
                if (roundCeiling == Long.MAX_VALUE) {
                    ++deltaToCeiling;
                }
                if ((diff = Longs.compare(deltaToFloor, deltaToCeiling)) < 0) {
                    return roundFloorAsDouble;
                }
                if (diff > 0) {
                    return roundCeilingAsDouble;
                }
                switch (mode) {
                    case HALF_EVEN: {
                        return (DoubleUtils.getSignificand(roundFloorAsDouble) & 1L) == 0L ? roundFloorAsDouble : roundCeilingAsDouble;
                    }
                    case HALF_DOWN: {
                        return x >= 0L ? roundFloorAsDouble : roundCeilingAsDouble;
                    }
                    case HALF_UP: {
                        return x >= 0L ? roundCeilingAsDouble : roundFloorAsDouble;
                    }
                }
                throw new AssertionError((Object)"impossible");
            }
        }
        throw new AssertionError((Object)"impossible");
    }

    private LongMath() {
    }

    private static enum MillerRabinTester {
        SMALL{

            @Override
            long mulMod(long a2, long b2, long m4) {
                return a2 * b2 % m4;
            }

            @Override
            long squareMod(long a2, long m4) {
                return a2 * a2 % m4;
            }
        }
        ,
        LARGE{

            private long plusMod(long a2, long b2, long m4) {
                return a2 >= m4 - b2 ? a2 + b2 - m4 : a2 + b2;
            }

            private long times2ToThe32Mod(long a2, long m4) {
                int shift;
                int remainingPowersOf2 = 32;
                do {
                    shift = Math.min(remainingPowersOf2, Long.numberOfLeadingZeros(a2));
                    a2 = UnsignedLongs.remainder(a2 << shift, m4);
                } while ((remainingPowersOf2 -= shift) > 0);
                return a2;
            }

            @Override
            long mulMod(long a2, long b2, long m4) {
                long aHi = a2 >>> 32;
                long bHi = b2 >>> 32;
                long aLo = a2 & 0xFFFFFFFFL;
                long bLo = b2 & 0xFFFFFFFFL;
                long result2 = this.times2ToThe32Mod(aHi * bHi, m4);
                if ((result2 += aHi * bLo) < 0L) {
                    result2 = UnsignedLongs.remainder(result2, m4);
                }
                result2 += aLo * bHi;
                result2 = this.times2ToThe32Mod(result2, m4);
                return this.plusMod(result2, UnsignedLongs.remainder(aLo * bLo, m4), m4);
            }

            @Override
            long squareMod(long a2, long m4) {
                long aHi = a2 >>> 32;
                long aLo = a2 & 0xFFFFFFFFL;
                long result2 = this.times2ToThe32Mod(aHi * aHi, m4);
                long hiLo = aHi * aLo * 2L;
                if (hiLo < 0L) {
                    hiLo = UnsignedLongs.remainder(hiLo, m4);
                }
                result2 += hiLo;
                result2 = this.times2ToThe32Mod(result2, m4);
                return this.plusMod(result2, UnsignedLongs.remainder(aLo * aLo, m4), m4);
            }
        };


        static boolean test(long base, long n) {
            return (n <= 3037000499L ? SMALL : LARGE).testWitness(base, n);
        }

        abstract long mulMod(long var1, long var3, long var5);

        abstract long squareMod(long var1, long var3);

        private long powMod(long a2, long p, long m4) {
            long res = 1L;
            while (p != 0L) {
                if ((p & 1L) != 0L) {
                    res = this.mulMod(res, a2, m4);
                }
                a2 = this.squareMod(a2, m4);
                p >>= 1;
            }
            return res;
        }

        private boolean testWitness(long base, long n) {
            int r = Long.numberOfTrailingZeros(n - 1L);
            long d2 = n - 1L >> r;
            if ((base %= n) == 0L) {
                return true;
            }
            long a2 = this.powMod(base, d2, n);
            if (a2 == 1L) {
                return true;
            }
            int j2 = 0;
            while (a2 != n - 1L) {
                if (++j2 == r) {
                    return false;
                }
                a2 = this.squareMod(a2, n);
            }
            return true;
        }
    }
}

