/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jgit.diff;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import org.eclipse.jgit.api.errors.CanceledException;
import org.eclipse.jgit.diff.ContentSource;
import org.eclipse.jgit.diff.DiffConfig;
import org.eclipse.jgit.diff.DiffEntry;
import org.eclipse.jgit.diff.SimilarityIndex;
import org.eclipse.jgit.diff.SimilarityRenameDetector;
import org.eclipse.jgit.internal.JGitText;
import org.eclipse.jgit.lib.AbbreviatedObjectId;
import org.eclipse.jgit.lib.FileMode;
import org.eclipse.jgit.lib.NullProgressMonitor;
import org.eclipse.jgit.lib.ObjectReader;
import org.eclipse.jgit.lib.ProgressMonitor;
import org.eclipse.jgit.lib.Repository;

public class RenameDetector {
    private static final int EXACT_RENAME_SCORE = 100;
    private static final Comparator<DiffEntry> DIFF_COMPARATOR = new Comparator<DiffEntry>(){

        @Override
        public int compare(DiffEntry a2, DiffEntry b2) {
            int cmp = this.nameOf(a2).compareTo(this.nameOf(b2));
            if (cmp == 0) {
                cmp = this.sortOf(a2.getChangeType()) - this.sortOf(b2.getChangeType());
            }
            return cmp;
        }

        private String nameOf(DiffEntry ent) {
            if (ent.changeType == DiffEntry.ChangeType.DELETE) {
                return ent.oldPath;
            }
            return ent.newPath;
        }

        private int sortOf(DiffEntry.ChangeType changeType) {
            switch (changeType) {
                case DELETE: {
                    return 1;
                }
                case ADD: {
                    return 2;
                }
            }
            return 10;
        }
    };
    private List<DiffEntry> entries;
    private List<DiffEntry> deleted;
    private List<DiffEntry> added;
    private boolean done;
    private final ObjectReader objectReader;
    private int renameScore = 60;
    private int breakScore = -1;
    private int renameLimit;
    private int bigFileThreshold = 0x3200000;
    private boolean skipContentRenamesForBinaryFiles = false;
    private boolean overRenameLimit;

    public RenameDetector(Repository repo) {
        this(repo.newObjectReader(), repo.getConfig().get(DiffConfig.KEY));
    }

    public RenameDetector(ObjectReader reader2, DiffConfig cfg) {
        this.objectReader = reader2.newReader();
        this.renameLimit = cfg.getRenameLimit();
        this.reset();
    }

    public int getRenameScore() {
        return this.renameScore;
    }

    public void setRenameScore(int score) {
        if (score < 0 || score > 100) {
            throw new IllegalArgumentException(JGitText.get().similarityScoreMustBeWithinBounds);
        }
        this.renameScore = score;
    }

    public int getBreakScore() {
        return this.breakScore;
    }

    public void setBreakScore(int breakScore) {
        this.breakScore = breakScore;
    }

    public int getRenameLimit() {
        return this.renameLimit;
    }

    public void setRenameLimit(int limit) {
        this.renameLimit = limit;
    }

    public int getBigFileThreshold() {
        return this.bigFileThreshold;
    }

    public void setBigFileThreshold(int threshold) {
        this.bigFileThreshold = threshold;
    }

    public boolean getSkipContentRenamesForBinaryFiles() {
        return this.skipContentRenamesForBinaryFiles;
    }

    public void setSkipContentRenamesForBinaryFiles(boolean value2) {
        this.skipContentRenamesForBinaryFiles = value2;
    }

    public boolean isOverRenameLimit() {
        if (this.done) {
            return this.overRenameLimit;
        }
        int cnt = Math.max(this.added.size(), this.deleted.size());
        return this.getRenameLimit() != 0 && this.getRenameLimit() < cnt;
    }

    public void addAll(Collection<DiffEntry> entriesToAdd) {
        if (this.done) {
            throw new IllegalStateException(JGitText.get().renamesAlreadyFound);
        }
        for (DiffEntry entry : entriesToAdd) {
            switch (entry.getChangeType()) {
                case ADD: {
                    this.added.add(entry);
                    break;
                }
                case DELETE: {
                    this.deleted.add(entry);
                    break;
                }
                case MODIFY: {
                    if (RenameDetector.sameType(entry.getOldMode(), entry.getNewMode())) {
                        this.entries.add(entry);
                        break;
                    }
                    List<DiffEntry> tmp = DiffEntry.breakModify(entry);
                    this.deleted.add(tmp.get(0));
                    this.added.add(tmp.get(1));
                    break;
                }
                default: {
                    this.entries.add(entry);
                }
            }
        }
    }

    public void add(DiffEntry entry) {
        this.addAll(Collections.singletonList(entry));
    }

    public List<DiffEntry> compute() throws IOException {
        try {
            return this.compute(NullProgressMonitor.INSTANCE);
        }
        catch (CanceledException e2) {
            return Collections.emptyList();
        }
    }

    public List<DiffEntry> compute(ProgressMonitor pm) throws IOException, CanceledException {
        if (!this.done) {
            try {
                List<DiffEntry> list = this.compute(this.objectReader, pm);
                return list;
            }
            finally {
                this.objectReader.close();
            }
        }
        return Collections.unmodifiableList(this.entries);
    }

    public List<DiffEntry> compute(ObjectReader reader2, ProgressMonitor pm) throws IOException, CanceledException {
        ContentSource cs = ContentSource.create(reader2);
        return this.compute(new ContentSource.Pair(cs, cs), pm);
    }

    public List<DiffEntry> compute(ContentSource.Pair reader2, ProgressMonitor pm) throws IOException, CanceledException {
        if (!this.done) {
            this.done = true;
            if (pm == null) {
                pm = NullProgressMonitor.INSTANCE;
            }
            if (this.breakScore > 0) {
                this.breakModifies(reader2, pm);
            }
            if (!this.added.isEmpty() && !this.deleted.isEmpty()) {
                this.findExactRenames(pm);
            }
            if (!this.added.isEmpty() && !this.deleted.isEmpty()) {
                this.findContentRenames(reader2, pm);
            }
            if (this.breakScore > 0 && !this.added.isEmpty() && !this.deleted.isEmpty()) {
                this.rejoinModifies(pm);
            }
            this.entries.addAll(this.added);
            this.added = null;
            this.entries.addAll(this.deleted);
            this.deleted = null;
            Collections.sort(this.entries, DIFF_COMPARATOR);
        }
        return Collections.unmodifiableList(this.entries);
    }

    public void reset() {
        this.entries = new ArrayList<DiffEntry>();
        this.deleted = new ArrayList<DiffEntry>();
        this.added = new ArrayList<DiffEntry>();
        this.done = false;
    }

    private void advanceOrCancel(ProgressMonitor pm) throws CanceledException {
        if (pm.isCancelled()) {
            throw new CanceledException(JGitText.get().renameCancelled);
        }
        pm.update(1);
    }

    private void breakModifies(ContentSource.Pair reader2, ProgressMonitor pm) throws IOException, CanceledException {
        ArrayList<DiffEntry> newEntries = new ArrayList<DiffEntry>(this.entries.size());
        pm.beginTask(JGitText.get().renamesBreakingModifies, this.entries.size());
        int i2 = 0;
        while (i2 < this.entries.size()) {
            DiffEntry e2 = this.entries.get(i2);
            if (e2.getChangeType() == DiffEntry.ChangeType.MODIFY) {
                int score = this.calculateModifyScore(reader2, e2);
                if (score < this.breakScore) {
                    List<DiffEntry> tmp = DiffEntry.breakModify(e2);
                    DiffEntry del = tmp.get(0);
                    del.score = score;
                    this.deleted.add(del);
                    this.added.add(tmp.get(1));
                } else {
                    newEntries.add(e2);
                }
            } else {
                newEntries.add(e2);
            }
            this.advanceOrCancel(pm);
            ++i2;
        }
        this.entries = newEntries;
    }

    private void rejoinModifies(ProgressMonitor pm) throws CanceledException {
        HashMap<String, DiffEntry> nameMap = new HashMap<String, DiffEntry>();
        ArrayList<DiffEntry> newAdded = new ArrayList<DiffEntry>(this.added.size());
        pm.beginTask(JGitText.get().renamesRejoiningModifies, this.added.size() + this.deleted.size());
        for (DiffEntry src : this.deleted) {
            nameMap.put(src.oldPath, src);
            this.advanceOrCancel(pm);
        }
        for (DiffEntry dst : this.added) {
            DiffEntry src = (DiffEntry)nameMap.remove(dst.newPath);
            if (src != null) {
                if (RenameDetector.sameType(src.oldMode, dst.newMode)) {
                    this.entries.add(DiffEntry.pair(DiffEntry.ChangeType.MODIFY, src, dst, src.score));
                } else {
                    nameMap.put(src.oldPath, src);
                    newAdded.add(dst);
                }
            } else {
                newAdded.add(dst);
            }
            this.advanceOrCancel(pm);
        }
        this.added = newAdded;
        this.deleted = new ArrayList(nameMap.values());
    }

    private int calculateModifyScore(ContentSource.Pair reader2, DiffEntry d2) throws IOException {
        try {
            SimilarityIndex src = new SimilarityIndex();
            src.hash(reader2.open(DiffEntry.Side.OLD, d2));
            src.sort();
            SimilarityIndex dst = new SimilarityIndex();
            dst.hash(reader2.open(DiffEntry.Side.NEW, d2));
            dst.sort();
            return src.score(dst, 100);
        }
        catch (SimilarityIndex.TableFullException tableFull) {
            this.overRenameLimit = true;
            return this.breakScore + 1;
        }
    }

    private void findContentRenames(ContentSource.Pair reader2, ProgressMonitor pm) throws IOException, CanceledException {
        int cnt = Math.max(this.added.size(), this.deleted.size());
        if (this.getRenameLimit() == 0 || cnt <= this.getRenameLimit()) {
            SimilarityRenameDetector d2 = new SimilarityRenameDetector(reader2, this.deleted, this.added);
            d2.setRenameScore(this.getRenameScore());
            d2.setBigFileThreshold(this.getBigFileThreshold());
            d2.setSkipBinaryFiles(this.getSkipContentRenamesForBinaryFiles());
            d2.compute(pm);
            this.overRenameLimit |= d2.isTableOverflow();
            this.deleted = d2.getLeftOverSources();
            this.added = d2.getLeftOverDestinations();
            this.entries.addAll(d2.getMatches());
        } else {
            this.overRenameLimit = true;
        }
    }

    private void findExactRenames(ProgressMonitor pm) throws CanceledException {
        DiffEntry best;
        pm.beginTask(JGitText.get().renamesFindingExact, this.added.size() + this.added.size() + this.deleted.size() + this.added.size() * this.deleted.size());
        HashMap<AbbreviatedObjectId, Object> deletedMap = this.populateMap(this.deleted, pm);
        HashMap<AbbreviatedObjectId, Object> addedMap = this.populateMap(this.added, pm);
        ArrayList<DiffEntry> uniqueAdds = new ArrayList<DiffEntry>(this.added.size());
        ArrayList<List> nonUniqueAdds = new ArrayList<List>();
        for (Object o : addedMap.values()) {
            if (o instanceof DiffEntry) {
                uniqueAdds.add((DiffEntry)o);
                continue;
            }
            nonUniqueAdds.add((List)o);
        }
        ArrayList<DiffEntry> left = new ArrayList<DiffEntry>(this.added.size());
        for (DiffEntry a2 : uniqueAdds) {
            Object del = deletedMap.get(a2.newId);
            if (del instanceof DiffEntry) {
                DiffEntry e2 = (DiffEntry)del;
                if (RenameDetector.sameType(e2.oldMode, a2.newMode)) {
                    e2.changeType = DiffEntry.ChangeType.RENAME;
                    this.entries.add(RenameDetector.exactRename(e2, a2));
                } else {
                    left.add(a2);
                }
            } else if (del != null) {
                List list = (List)del;
                best = RenameDetector.bestPathMatch(a2, list);
                if (best != null) {
                    best.changeType = DiffEntry.ChangeType.RENAME;
                    this.entries.add(RenameDetector.exactRename(best, a2));
                } else {
                    left.add(a2);
                }
            } else {
                left.add(a2);
            }
            this.advanceOrCancel(pm);
        }
        for (List adds : nonUniqueAdds) {
            Object o = deletedMap.get(((DiffEntry)adds.get((int)0)).newId);
            if (o instanceof DiffEntry) {
                DiffEntry d2 = (DiffEntry)o;
                best = RenameDetector.bestPathMatch(d2, adds);
                if (best != null) {
                    d2.changeType = DiffEntry.ChangeType.RENAME;
                    this.entries.add(RenameDetector.exactRename(d2, best));
                    for (DiffEntry a3 : adds) {
                        if (a3 == best) continue;
                        if (RenameDetector.sameType(d2.oldMode, a3.newMode)) {
                            this.entries.add(RenameDetector.exactCopy(d2, a3));
                            continue;
                        }
                        left.add(a3);
                    }
                } else {
                    left.addAll(adds);
                }
            } else if (o != null) {
                List dels = (List)o;
                long[] matrix = new long[dels.size() * adds.size()];
                int mNext = 0;
                int delIdx = 0;
                while (delIdx < dels.size()) {
                    String deletedName = ((DiffEntry)dels.get((int)delIdx)).oldPath;
                    int addIdx = 0;
                    while (addIdx < adds.size()) {
                        String addedName = ((DiffEntry)adds.get((int)addIdx)).newPath;
                        int score = SimilarityRenameDetector.nameScore(addedName, deletedName);
                        matrix[mNext] = SimilarityRenameDetector.encode(score, delIdx, addIdx);
                        ++mNext;
                        if (pm.isCancelled()) {
                            throw new CanceledException(JGitText.get().renameCancelled);
                        }
                        ++addIdx;
                    }
                    ++delIdx;
                }
                Arrays.sort(matrix);
                --mNext;
                while (mNext >= 0) {
                    long ent = matrix[mNext];
                    int delIdx2 = SimilarityRenameDetector.srcFile(ent);
                    int addIdx = SimilarityRenameDetector.dstFile(ent);
                    DiffEntry d3 = (DiffEntry)dels.get(delIdx2);
                    DiffEntry a4 = (DiffEntry)adds.get(addIdx);
                    if (a4 == null) {
                        this.advanceOrCancel(pm);
                    } else {
                        DiffEntry.ChangeType type2;
                        if (d3.changeType == DiffEntry.ChangeType.DELETE) {
                            d3.changeType = DiffEntry.ChangeType.RENAME;
                            type2 = DiffEntry.ChangeType.RENAME;
                        } else {
                            type2 = DiffEntry.ChangeType.COPY;
                        }
                        this.entries.add(DiffEntry.pair(type2, d3, a4, 100));
                        adds.set(addIdx, null);
                        this.advanceOrCancel(pm);
                    }
                    --mNext;
                }
            } else {
                left.addAll(adds);
            }
            this.advanceOrCancel(pm);
        }
        this.added = left;
        this.deleted = new ArrayList<DiffEntry>(deletedMap.size());
        for (Object o : deletedMap.values()) {
            if (o instanceof DiffEntry) {
                DiffEntry e3 = (DiffEntry)o;
                if (e3.changeType != DiffEntry.ChangeType.DELETE) continue;
                this.deleted.add(e3);
                continue;
            }
            List list = (List)o;
            for (DiffEntry e2 : list) {
                if (e2.changeType != DiffEntry.ChangeType.DELETE) continue;
                this.deleted.add(e2);
            }
        }
        pm.endTask();
    }

    private static DiffEntry bestPathMatch(DiffEntry src, List<DiffEntry> list) {
        DiffEntry best = null;
        int score = -1;
        for (DiffEntry d2 : list) {
            int tmp;
            if (!RenameDetector.sameType(RenameDetector.mode(d2), RenameDetector.mode(src)) || (tmp = SimilarityRenameDetector.nameScore(RenameDetector.path(d2), RenameDetector.path(src))) <= score) continue;
            best = d2;
            score = tmp;
        }
        return best;
    }

    private HashMap<AbbreviatedObjectId, Object> populateMap(List<DiffEntry> diffEntries, ProgressMonitor pm) throws CanceledException {
        HashMap<AbbreviatedObjectId, Object> map2 = new HashMap<AbbreviatedObjectId, Object>();
        for (DiffEntry de : diffEntries) {
            Object old = map2.put(RenameDetector.id(de), de);
            if (old instanceof DiffEntry) {
                ArrayList<DiffEntry> list = new ArrayList<DiffEntry>(2);
                list.add((DiffEntry)old);
                list.add(de);
                map2.put(RenameDetector.id(de), list);
            } else if (old != null) {
                ((List)old).add(de);
                map2.put(RenameDetector.id(de), old);
            }
            this.advanceOrCancel(pm);
        }
        return map2;
    }

    private static String path(DiffEntry de) {
        return de.changeType == DiffEntry.ChangeType.DELETE ? de.oldPath : de.newPath;
    }

    private static FileMode mode(DiffEntry de) {
        return de.changeType == DiffEntry.ChangeType.DELETE ? de.oldMode : de.newMode;
    }

    private static AbbreviatedObjectId id(DiffEntry de) {
        return de.changeType == DiffEntry.ChangeType.DELETE ? de.oldId : de.newId;
    }

    static boolean sameType(FileMode a2, FileMode b2) {
        int bType;
        int aType = a2.getBits() & 0xF000;
        return aType == (bType = b2.getBits() & 0xF000);
    }

    private static DiffEntry exactRename(DiffEntry src, DiffEntry dst) {
        return DiffEntry.pair(DiffEntry.ChangeType.RENAME, src, dst, 100);
    }

    private static DiffEntry exactCopy(DiffEntry src, DiffEntry dst) {
        return DiffEntry.pair(DiffEntry.ChangeType.COPY, src, dst, 100);
    }
}

