/*
 * Decompiled with CFR 0.152.
 */
package org.thymeleaf.cache;

import java.lang.ref.SoftReference;
import java.util.Arrays;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import org.slf4j.Logger;
import org.thymeleaf.TemplateEngine;
import org.thymeleaf.cache.ICache;
import org.thymeleaf.cache.ICacheEntryValidityChecker;
import org.thymeleaf.util.Validate;

public final class StandardCache<K, V>
implements ICache<K, V> {
    private static final long REPORT_INTERVAL = 300000L;
    private static final String REPORT_FORMAT = "[THYMELEAF][*][*][*][CACHE_REPORT] %8s elements | %12s puts | %12s gets | %12s hits | %12s misses | %.2f hit ratio | %.2f miss ratio - [%s]";
    private volatile long lastExecution = System.currentTimeMillis();
    private final String name;
    private final boolean useSoftReferences;
    private final int maxSize;
    private final CacheDataContainer<K, V> dataContainer;
    private final ICacheEntryValidityChecker<? super K, ? super V> entryValidityChecker;
    private final boolean traceExecution;
    private final boolean enableCounters;
    private final Logger logger;
    private final AtomicLong getCount;
    private final AtomicLong putCount;
    private final AtomicLong hitCount;
    private final AtomicLong missCount;

    public StandardCache(String name, boolean useSoftReferences, int initialCapacity, Logger logger) {
        this(name, useSoftReferences, initialCapacity, -1, null, logger, false);
    }

    public StandardCache(String name, boolean useSoftReferences, int initialCapacity, ICacheEntryValidityChecker<? super K, ? super V> entryValidityChecker, Logger logger) {
        this(name, useSoftReferences, initialCapacity, -1, entryValidityChecker, logger, false);
    }

    public StandardCache(String name, boolean useSoftReferences, int initialCapacity, int maxSize2, Logger logger) {
        this(name, useSoftReferences, initialCapacity, maxSize2, null, logger, false);
    }

    public StandardCache(String name, boolean useSoftReferences, int initialCapacity, int maxSize2, ICacheEntryValidityChecker<? super K, ? super V> entryValidityChecker, Logger logger) {
        this(name, useSoftReferences, initialCapacity, maxSize2, entryValidityChecker, logger, false);
    }

    public StandardCache(String name, boolean useSoftReferences, int initialCapacity, int maxSize2, ICacheEntryValidityChecker<? super K, ? super V> entryValidityChecker, Logger logger, boolean enableCounters) {
        Validate.notEmpty(name, "Name cannot be null or empty");
        Validate.isTrue(initialCapacity > 0, "Initial capacity must be > 0");
        Validate.isTrue(maxSize2 != 0, "Cache max size must be either -1 (no limit) or > 0");
        this.name = name;
        this.useSoftReferences = useSoftReferences;
        this.maxSize = maxSize2;
        this.entryValidityChecker = entryValidityChecker;
        this.logger = logger;
        this.traceExecution = logger != null && logger.isTraceEnabled();
        this.enableCounters = this.traceExecution || enableCounters;
        this.dataContainer = new CacheDataContainer(this.name, initialCapacity, maxSize2, this.traceExecution, this.logger);
        this.getCount = new AtomicLong(0L);
        this.putCount = new AtomicLong(0L);
        this.hitCount = new AtomicLong(0L);
        this.missCount = new AtomicLong(0L);
        if (this.logger != null) {
            if (this.maxSize < 0) {
                this.logger.trace("[THYMELEAF][CACHE_INITIALIZE] Initializing cache {}. Soft references {}.", (Object)this.name, (Object)(this.useSoftReferences ? "are used" : "not used"));
            } else {
                this.logger.trace("[THYMELEAF][CACHE_INITIALIZE] Initializing cache {}. Max size: {}. Soft references {}.", this.name, this.maxSize, this.useSoftReferences ? "are used" : "not used");
            }
        }
    }

    @Override
    public void put(K key2, V value2) {
        this.incrementReportEntity(this.putCount);
        CacheEntry<V> entry = new CacheEntry<V>(value2, this.useSoftReferences);
        int newSize = this.dataContainer.put(key2, entry);
        if (this.traceExecution) {
            this.logger.trace("[THYMELEAF][{}][{}][CACHE_ADD][{}] Adding cache entry in cache \"{}\" for key \"{}\". New size is {}.", TemplateEngine.threadIndex(), this.name, newSize, this.name, key2, newSize);
            this.outputReportIfNeeded();
        }
    }

    @Override
    public V get(K key2) {
        return this.get(key2, this.entryValidityChecker);
    }

    @Override
    public V get(K key2, ICacheEntryValidityChecker<? super K, ? super V> validityChecker) {
        this.incrementReportEntity(this.getCount);
        CacheEntry<? super V> resultEntry = this.dataContainer.get(key2);
        if (resultEntry == null) {
            this.incrementReportEntity(this.missCount);
            if (this.traceExecution) {
                this.logger.trace("[THYMELEAF][{}][{}][CACHE_MISS] Cache miss in cache \"{}\" for key \"{}\".", TemplateEngine.threadIndex(), this.name, this.name, key2);
                this.outputReportIfNeeded();
            }
            return null;
        }
        V resultValue = resultEntry.getValueIfStillValid(this.name, key2, validityChecker, this.traceExecution, this.logger);
        if (resultValue == null) {
            int newSize = this.dataContainer.remove(key2);
            this.incrementReportEntity(this.missCount);
            if (this.traceExecution) {
                this.logger.trace("[THYMELEAF][{}][{}][CACHE_REMOVE][{}] Removing cache entry in cache \"{}\" (Entry \"{}\" is not valid anymore). New size is {}.", TemplateEngine.threadIndex(), this.name, newSize, this.name, key2, newSize);
                this.logger.trace("[THYMELEAF][{}][{}][CACHE_MISS] Cache miss in cache \"{}\" for key \"{}\".", TemplateEngine.threadIndex(), this.name, this.name, key2);
                this.outputReportIfNeeded();
            }
            return null;
        }
        this.incrementReportEntity(this.hitCount);
        if (this.traceExecution) {
            this.logger.trace("[THYMELEAF][{}][{}][CACHE_HIT] Cache hit in cache \"{}\" for key \"{}\".", TemplateEngine.threadIndex(), this.name, this.name, key2);
            this.outputReportIfNeeded();
        }
        return resultValue;
    }

    @Override
    public Set<K> keySet() {
        return this.dataContainer.keySet();
    }

    @Override
    public void clear() {
        this.dataContainer.clear();
        if (this.traceExecution) {
            this.logger.trace("[THYMELEAF][{}][*][{}][CACHE_REMOVE][0] Removing ALL cache entries in cache \"{}\". New size is 0.", TemplateEngine.threadIndex(), this.name, this.name);
        }
    }

    @Override
    public void clearKey(K key2) {
        int newSize = this.dataContainer.remove(key2);
        if (this.traceExecution && newSize != -1) {
            this.logger.trace("[THYMELEAF][{}][*][{}][CACHE_REMOVE][{}] Removed cache entry in cache \"{}\" for key \"{}\". New size is {}.", TemplateEngine.threadIndex(), this.name, newSize, this.name, key2, newSize);
        }
    }

    public String getName() {
        return this.name;
    }

    public boolean hasMaxSize() {
        return this.maxSize > 0;
    }

    public int getMaxSize() {
        return this.maxSize;
    }

    public boolean getUseSoftReferences() {
        return this.useSoftReferences;
    }

    public int size() {
        return this.dataContainer.size();
    }

    public long getPutCount() {
        return this.putCount.get();
    }

    public long getGetCount() {
        return this.getCount.get();
    }

    public long getHitCount() {
        return this.hitCount.get();
    }

    public long getMissCount() {
        return this.missCount.get();
    }

    public double getHitRatio() {
        long hitCount = this.getHitCount();
        long getCount = this.getGetCount();
        if (hitCount == 0L || getCount == 0L) {
            return 0.0;
        }
        return (double)hitCount / (double)getCount;
    }

    public double getMissRatio() {
        return 1.0 - this.getHitRatio();
    }

    private void incrementReportEntity(AtomicLong entity) {
        if (this.enableCounters) {
            entity.incrementAndGet();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void outputReportIfNeeded() {
        long currentTime = System.currentTimeMillis();
        if (currentTime - this.lastExecution >= 300000L) {
            StandardCache standardCache = this;
            synchronized (standardCache) {
                if (currentTime - this.lastExecution >= 300000L) {
                    long hitCount = this.getHitCount();
                    long missCount = this.getMissCount();
                    long putCount = this.getPutCount();
                    long getCount = this.getGetCount();
                    double hitRatio = (double)hitCount / (double)getCount;
                    double missRatio = 1.0 - hitRatio;
                    this.logger.trace(String.format(REPORT_FORMAT, this.size(), putCount, getCount, hitCount, missCount, hitRatio, missRatio, this.name));
                    this.lastExecution = currentTime;
                }
            }
        }
    }

    static final class CacheDataContainer<K, V> {
        private final String name;
        private final boolean sizeLimit;
        private final int maxSize;
        private final boolean traceExecution;
        private final Logger logger;
        private final ConcurrentHashMap<K, CacheEntry<V>> container;
        private final Object[] fifo;
        private int fifoPointer;

        CacheDataContainer(String name, int initialCapacity, int maxSize2, boolean traceExecution, Logger logger) {
            this.name = name;
            this.container = new ConcurrentHashMap(initialCapacity, 0.9f, 2);
            this.maxSize = maxSize2;
            boolean bl = this.sizeLimit = maxSize2 >= 0;
            if (this.sizeLimit) {
                this.fifo = new Object[this.maxSize];
                Arrays.fill(this.fifo, null);
            } else {
                this.fifo = null;
            }
            this.fifoPointer = 0;
            this.traceExecution = traceExecution;
            this.logger = logger;
        }

        public CacheEntry<V> get(Object key2) {
            return this.container.get(key2);
        }

        public Set<K> keySet() {
            return this.container.keySet();
        }

        public int put(K key2, CacheEntry<V> value2) {
            if (this.traceExecution) {
                return this.putWithTracing(key2, value2);
            }
            return this.putWithoutTracing(key2, value2);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         * Enabled force condition propagation
         * Lifted jumps to return sites
         */
        private int putWithoutTracing(K key2, CacheEntry<V> value2) {
            CacheEntry<V> existing = this.container.putIfAbsent(key2, value2);
            if (existing != null) {
                return -1;
            }
            if (!this.sizeLimit) return -1;
            Object[] objectArray = this.fifo;
            synchronized (this.fifo) {
                Object removedKey = this.fifo[this.fifoPointer];
                if (removedKey != null) {
                    this.container.remove(removedKey);
                }
                this.fifo[this.fifoPointer] = key2;
                this.fifoPointer = (this.fifoPointer + 1) % this.maxSize;
                // ** MonitorExit[var4_4] (shouldn't be in output)
                return -1;
            }
        }

        private synchronized int putWithTracing(K key2, CacheEntry<V> value2) {
            CacheEntry<V> existing = this.container.putIfAbsent(key2, value2);
            if (existing == null && this.sizeLimit) {
                CacheEntry<V> removed;
                Object removedKey = this.fifo[this.fifoPointer];
                if (removedKey != null && (removed = this.container.remove(removedKey)) != null) {
                    Integer newSize = this.container.size();
                    this.logger.trace("[THYMELEAF][{}][{}][CACHE_REMOVE][{}] Max size exceeded for cache \"{}\". Removing entry for key \"{}\". New size is {}.", TemplateEngine.threadIndex(), this.name, newSize, this.name, removedKey, newSize);
                }
                this.fifo[this.fifoPointer] = key2;
                this.fifoPointer = (this.fifoPointer + 1) % this.maxSize;
            }
            return this.container.size();
        }

        public int remove(K key2) {
            if (this.traceExecution) {
                return this.removeWithTracing(key2);
            }
            return this.removeWithoutTracing(key2);
        }

        private int removeWithoutTracing(K key2) {
            CacheEntry<V> removed = this.container.remove(key2);
            if (removed != null && this.sizeLimit && key2 != null) {
                for (int i2 = 0; i2 < this.maxSize; ++i2) {
                    if (!key2.equals(this.fifo[i2])) continue;
                    this.fifo[i2] = null;
                    break;
                }
            }
            return -1;
        }

        private synchronized int removeWithTracing(K key2) {
            CacheEntry<V> removed = this.container.remove(key2);
            if (removed == null) {
                return -1;
            }
            if (this.sizeLimit && key2 != null) {
                for (int i2 = 0; i2 < this.maxSize; ++i2) {
                    if (!key2.equals(this.fifo[i2])) continue;
                    this.fifo[i2] = null;
                    break;
                }
            }
            return this.container.size();
        }

        public void clear() {
            this.container.clear();
        }

        public int size() {
            return this.container.size();
        }
    }

    static final class CacheEntry<V> {
        private final SoftReference<V> cachedValueReference;
        private final long creationTimeInMillis;
        private final V cachedValueAnchor;

        CacheEntry(V cachedValue, boolean useSoftReferences) {
            this.cachedValueReference = new SoftReference<V>(cachedValue);
            this.cachedValueAnchor = !useSoftReferences ? cachedValue : null;
            this.creationTimeInMillis = System.currentTimeMillis();
        }

        public <K> V getValueIfStillValid(String cacheMapName, K key2, ICacheEntryValidityChecker<? super K, ? super V> checker, boolean traceExecution, Logger logger) {
            V cachedValue = this.cachedValueReference.get();
            if (cachedValue == null) {
                if (traceExecution) {
                    logger.trace("[THYMELEAF][{}][*][{}][CACHE_DELETED_REFERENCES] Some entries at cache \"{}\" seem to have been sacrificed by the Garbage Collector (soft references).", TemplateEngine.threadIndex(), cacheMapName, cacheMapName);
                }
                return null;
            }
            if (checker == null || checker.checkIsValueStillValid(key2, cachedValue, this.creationTimeInMillis)) {
                return cachedValue;
            }
            return null;
        }

        public long getCreationTimeInMillis() {
            return this.creationTimeInMillis;
        }
    }
}

