/**
 * @typedef {"success" | "failed" | "error" | "notcovered" | "total"} ResultType
 */

class TableFilter {
    /**
     * @param {Object} options
     * @param {HTMLOListElement} options.countsList
     * @param {Map<string, HTMLElement>} options.countsMap
     * @param {TableMaker} options.tableMaker
     * @param {(arg: { group: string, coverage: number }) => void} options.updateGroupCoverage
     */
    constructor({ countsList, countsMap, tableMaker, updateGroupCoverage }) {
        this.countsList = countsList;
        this.countsMap = countsMap;
        this.tableMaker = tableMaker;
        this.updateGroupCoverage = updateGroupCoverage;

        this._events = [];
        /** @type {Map<string, number>} */
        this._coverage = new Map();
        /** @type {ResultType|null} */
        this.savedFilter = null;
        this.countsList.addEventListener('click', this._onClick.bind(this));
    }

    /** @param {MouseEvent} event */
    _onClick(event) {
        const li = event.target?.closest('li');
        if (!li) return;

        const type = li.getAttribute("data-type");
        const count = Number(li.querySelector('span:last-child')?.getAttribute("data-value") || "0");
        if (!type || count === 0) return createAlert({ title: "Cannot Filter", message: "No data to filter on", type: "info", duration: 1500 });

        const isReset = type === "total" || this.countsList.getAttribute("data-filter") === type;
        this._applyFilter(isReset ? "total" : type);
    }

    /**
     * @param {ResultType} type
     */
    _applyFilter(type) {
        const li = this.countsList.querySelector(`[data-type="${type}"]`);
        for (const el of this.countsList.querySelectorAll('[data-active]')) el.removeAttribute('data-active');
        if (li) li.setAttribute('data-active', 'true');
        this.countsList.setAttribute("data-filter", type);

        const rows = this.getFilteredEvents(type);
        requestAnimationFrame(() => {
            this.tableMaker.reset();
            for (const row of rows) this.tableMaker.addRow(row, false);
            this._updatePaths(rows);
        });
    }

    /**
     * @param {ResultType} type
     * @returns {Object[]}
     */
    getFilteredEvents(type) {
        return type === "total" ? this._events : this._events.filter(e => e.__normalizedResult === type);
    }

    /**
     * @param {Object} data
     */
    recordAndAddRow(data) {
        const result = this._normalize(data._result);
        this._updateIfExistsElseAdd({ data, result });

        const current = this.countsList.getAttribute("data-filter");
        if (current !== "total" && current !== result) {
            updateCounts({ countsMap: this.countsMap, result: data._result, by: data.result || 1 });
            return;
        }

        this.tableMaker.addRow(data);
    }

    /**
     * Build a stable id for grouping rows across placeholder and events.
     * @param {Object} e
     * @returns {string}
     */
    _getId(e) {
        return this.tableMaker.getKeyFor(e);
    }

    _updateIfExistsElseAdd({data, result}) {
        const dataId = this._getId(data);
        const undefinedEntry = this._events.find(e => !e.remark && !e.result && this._getId(e) === dataId);
        if (undefinedEntry) {
            undefinedEntry.remark = data.remark;
            undefinedEntry.result = data.result;
            undefinedEntry._result = data._result;
            undefinedEntry.__skipRowResult = false;
            undefinedEntry.__normalizedResult = result;
            return;
        }

        this._events.push({ ...data, __normalizedResult: result, __skipCount: true });
    }

    /**
     * @param {string|undefined} r
     * @returns {ResultType}
     */
    _normalize(r) {
        const v = (r || "notcovered").toLowerCase();
        return ["success", "error", "notcovered"].includes(v) ? /** @type {ResultType} */ (v) : "failed";
    }

    /**
     * @param {Object[]} events
     */
    _updatePaths(events) {
        const groups = new Set(events.map(e => this.tableMaker.getMainGroupEntry(e)));
        for (const { value } of groups) {
            const coverage = this._coverage.get(value);
            if (coverage === undefined) continue;
            this.updateGroupCoverage({ group: value, coverage });
        }
    }

    /**

     * @param {string} group
     * @param {number} coverage
     */
    addCoverage(group, coverage) {
        this._coverage.set(group, coverage);
        try { this.updateGroupCoverage({ group, coverage }) } catch(e) { console.log(e) /* Can fail if path not on screen  */ }
    }

    getCurrentFilter() {
        const value = this.countsList.getAttribute("data-filter") || "total";
        return /** @type {ResultType} */ (value);
    }

    reset(tableMaker) {
        const saved = this.countsList.getAttribute("data-filter");
        this.savedFilter = saved !== "total" ? /** @type {ResultType|null} */ (saved) : null;

        for (const el of this.countsList.querySelectorAll('[data-active]')) el.removeAttribute('data-active');
        this.countsList.setAttribute("data-filter", "total");
        this.countsList.querySelector('li[data-type="total"]')?.setAttribute('data-active', 'true');

        this._events = [];
        this._coverage.clear();
        this.tableMaker = tableMaker;
    }

    onEnd(applyLastFilter) {
        const notCovered = this._events.filter(e => !e.result && !e.remark);
        const currentFilter = this.getCurrentFilter();
        for (const e of notCovered) Object.assign(e, { remark: "notcovered", result: 1, _result: "notcovered", __skipRowResult: false });
        if (currentFilter !== "total" && currentFilter !== "notcovered") updateCounts({ result: "notcovered", countsMap: this.countsMap, by: notCovered.length });
        if (this.savedFilter && applyLastFilter) this._applyFilter(this.savedFilter);
    }

    removeExcludedPath(path) {
        this._events = this._events.filter(e => e.path !== path);
        this._coverage.delete(path);
        updateCounts({ countsMap: this.countsMap, result: "notcovered", by: -1 });
    }
}
