const panzoomMap = new Map();

document.addEventListener('DOMContentLoaded', () => {
    console.log('Mermaid Loaded');
    mermaid.initialize({ startOnLoad: false });
});

function wheelHandler(event, panzoomInstance) {
    if (!event.shiftKey) return;
    panzoomInstance.zoomWithWheel(event);
}

async function renderMermaidDiagram(container, diagramData, resize = true, panZoom = true) {
    container.innerHTML = "";
    container.removeAttribute('data-processed');
    container.textContent = diagramData;
    if (resize) resizeDiagram(container);

    if (panzoomMap.has(container)) {
        const { panzoomInstance, wheelHandler } = panzoomMap.get(container);
        panzoomInstance.destroy?.();
        container.removeEventListener("wheel", wheelHandler);
        panzoomMap.delete(container);
    }

    await mermaid.run({
        nodes: [container],
        postRenderCallback: (id) => {
            if (!panZoom) return;
            const svgElement = document.getElementById(id);
            if (!svgElement) return;
            const panzoomInstance = Panzoom(svgElement);

            const wheelHandler = (event) => {
                if (!event.shiftKey) return;
                panzoomInstance.zoomWithWheel(event);
            };
            svgElement.parentElement.addEventListener("wheel", wheelHandler);

            let scaleFactor = 1;
            svgElement.addEventListener("gesturestart", (event) => {
                event.preventDefault();
                scaleFactor = panzoomInstance.getScale();
            });

            svgElement.addEventListener("gesturechange", (event) => {
                event.preventDefault();
                const newScale = scaleFactor * event.scale;
                panzoomInstance.zoomToPoint(newScale, event);
            });

            panzoomMap.set(container, { panzoomInstance, wheelHandler });
        }
    });
}

function resizeDiagram(container) {
    container.style.maxHeight = `${window.innerHeight - 200}px`;
}

class MermaidDrillDown {
    constructor({ eventId, container }) {
        this.eventId = eventId;
        this.container = container;
        this.mainDiv = container.parentElement;
        if (!this.mainDiv) throw Error("No main div found");
        this.selector = container.querySelector("select#diagramSelector");
        this.diagramsContainer = container.querySelector(".diagramContainer");
        const legend = container.querySelector(".mermaid > .legend");
        renderMermaidDiagram(legend, legend.textContent, false, false);
        this._addEventListeners();
    }

    reset(eventId, fetchNow = true) {
        this.eventId = eventId;
        this.selector.innerHTML = "";
        this.diagramsContainer.innerHTML = "";
        if (fetchNow) this._retrieveContexts()
        this.#listenForDiagramEvents();
    }

    _addEventListeners() {
        this.mainDiv.addEventListener("click", async (e) => {
            const flowChartButton = e.target.closest(".flow-chart-btn");
            const nearestBackBtn = e.target.closest(".back-btn");
            if (!flowChartButton && !nearestBackBtn) return;
            this.mainDiv.setAttribute("data-mode", flowChartButton ? "diagrams" : "table");
            e.stopPropagation();
        });

        this.selector.addEventListener("change", async (e) => {
            const contextName = e.target.value;
            if (!contextName) return;
            const diagram = this.diagramsContainer.querySelector(`[data-context-name="${contextName}"]`);
            if (!diagram) return;
            this.diagramsContainer.querySelectorAll(".mermaid").forEach((diagram) => diagram.classList.add("hidden"));
            diagram.classList.remove("hidden");
            if (diagram.getAttribute("render") === "true") {
                await this._renderDiagram(diagram, diagram.textContent);
                diagram.removeAttribute("render");
            };
        });
    }

    async _retrieveContexts() {
        const { data: contextData, error } = await makeHttpCall(`/workflow/${this.eventId}/contexts`, { method: "GET" });
        if (error) {
            createAlert({ title: "Failed to get workflow contexts", message: error, type: "error" });
            throw new Error(error);
        }

        for (const { name, data } of contextData) {
            this._addOptionToSelector(name);
            let diagramDiv = this.diagramsContainer.querySelector(`[data-context-name="${name}"]`);
            if (!diagramDiv) diagramDiv = this._createDiagramDiv(name);
            await this._renderDiagram(diagramDiv, data);
        }

        const diagrams = this.diagramsContainer.querySelectorAll(".mermaid");
        for (const diagram of diagrams) {
            const contextName = diagram.getAttribute("data-context-name");
            diagram.classList.toggle("hidden", contextName !== "DEFAULT" && contextName !== "Flow1");
        }
    }

    async refreshDiagram({ name, diagram }) {
        let diagramDiv = this.diagramsContainer.querySelector(`[data-context-name="${name}"]`);
        if (!diagramDiv) {
            diagramDiv = this._createDiagramDiv(name);
            this._addOptionToSelector(name);
        }

        if (diagramDiv.classList.contains("hidden")) {
            diagramDiv.textContent = diagram;
            diagramDiv.setAttribute("render", "true");
            return;
        };

        await this._renderDiagram(diagramDiv, diagram);
        const shouldHideByDefault = name !== "DEFAULT" && name !== "Flow1";
        diagramDiv.classList.toggle("hidden", shouldHideByDefault);
    }

    _createDiagramDiv(name) {
        const diagramDiv = document.createElement("div");
        diagramDiv.classList.add("mermaid");
        diagramDiv.setAttribute("data-context-name", name);
        this.diagramsContainer.appendChild(diagramDiv);
        return diagramDiv;
    }

    _addOptionToSelector(name) {
        const existingOption = this.selector.querySelector(`option[value="${name}"]`);
        if (!existingOption) {
            const option = document.createElement("option");
            option.value = name;
            option.textContent = name;
            this.selector.appendChild(option);
        }
    }

    async _renderDiagram(diagramDiv, diagram) {
        try {
            await renderMermaidDiagram(diagramDiv, diagram, false, true);
        } catch (error) {
            console.error(`Failed to render diagram ${diagramDiv.getAttribute("data-context-name")}:`, error);
        }
    }

    #listenForDiagramEvents() {
        SseEventStreamer.subscribe({
            eventId: this.eventId,
            callbacks: {
                onData: (data) => {
                    if (data.type !== "ArazzoMockDiagramEvent") return;
                    this.refreshDiagram({ name: data.name, diagram: data.diagram });
                }
            }
        })
    }
}
