/** @type {Map<HTMLElement, TableMaker>} */
const screenToMockTableMaker = new Map();
/** @type {Map<HTMLElement, DrillDown>} */
const screenToMockDrillDown = new Map();
/** @type {Map<HTMLElement, TableFilter>} */
const screenToMockTableFilter = new Map();
/** @type {Map<HTMLElement, String>} */
const screenToEventId = new Map();
/** @type {Map<HTMLElement, MermaidDrillDown>} */
const screenToMockMermaidDrillDown = new Map();

const mockEventSubscriptions = [];
let keepSubscribingToMockEvents = false;

const MOCK_INFO_SELECTOR = ".details > .mock"
const fileTypeToGroupByDetails = new Map([
    ["openapi", new Set(["path", "method", "response"])],
    ["asyncapi", new Set(["channel", "action"])],
    ["arazzo", new Set(["workflow", "step"])]
]);

function clearMockFor(screen) {
    if (!screenToMockTableMaker.has(screen)) return;
    screenToMockTableMaker.delete(screen);
    initMock(screen);
    const drilldown = screenToMockDrillDown.get(screen);
    const eventId = screenToEventId.get(screen);
    if (drilldown && eventId) {
        drilldown.updateEventId(eventId);
    }
}

function registerListenerForMock(screen) {
    subscribeToMockEvents(screen);
}

function initMock(screen) {
    subscribeToMockEvents(screen)
    const fileType = screen.getAttribute('data-file-type');
    const groupBy = fileTypeToGroupByDetails.get(fileType);
    if (!groupBy) throw new Error(`Unsupported file type: ${fileType}`);
    init({
        screen,
        tab: screen.querySelector(".details .mock"),
        withSelection: false,
        tableStore: screenToMockTableMaker,
        filterStore: screenToMockTableFilter,
        groupBy: groupBy
    });
}

async function handleMock(screen) {
    const btn = screen.querySelector(".details .mock button.run");
    const path = screen.id;
    switch (btn.getAttribute("data-running")) {
        case "true": await stopMock(screen); break;
        case "false": await runMock(screen, path); break;
        default: return;
    }
}

async function runMock(screen, path) {
    keepSubscribingToMockEvents = true;
    subscribeToMockEvents(screen);
    screen.querySelector(".details .mock button.run").setAttribute("data-running", "processing");
    const fileType = screen.getAttribute('data-file-type');
    const groupBy = fileTypeToGroupByDetails.get(fileType);
    if (!groupBy) throw new Error(`Unsupported file type: ${fileType}`);
    run({
        screen,
        tab: screen.querySelector(".details .mock"),
        tableStore: screenToMockTableMaker,
        drillDownStore: screenToMockDrillDown,
        filterStore: screenToMockTableFilter,
        needStart: false,
        getEventId: async () => {
            if (fileType === "arazzo") return await getWorkflowMockEventId(screen, path);
            else return await getMockEventId(screen, path, screen.querySelector(".details .mock #mockPort"))
        },
        onDrillDown: getMockDetails,
        groupBy: groupBy
    })
}

function unsubscribeMockEvents() {
    keepSubscribingToMockEvents = false;
    try {
        mockEventSubscriptions.forEach(unsubscribe => unsubscribe());
    } catch (e) {
        console.log("An error occurred while unsubscribing the mock event: " + e);
    }
}

async function stopMock(screen) {
    unsubscribeMockEvents();
    screen.querySelector(".details .mock button.run").setAttribute("data-running", "processing");
    const eventId = screenToEventId.get(screen);
    const { error } = await makeHttpCall("/mock/stop", { method: "POST", body: { id: eventId } });
    if (error) {
        createAlert({ title: "Failed to stop mock server", message: JSON.stringify(error), type: "error" });
        Workspace.updateInfoBox(screen, "title" + "\n" + JSON.stringify(error), MOCK_INFO_SELECTOR);
    }
    screenToEventId.delete(screen);
    if(!error) {
        showPreReqErrorBar(null, screen);
        Workspace.updateInfoBox(screen, null, MOCK_INFO_SELECTOR);
    }
}

function stopAllMocks() {
    for (const eventId of screenToEventId.values()) {
        makeHttpCall("/mock/stop", { method: "POST", body: { id: eventId } });
    }
}
window.addEventListener('unload', stopAllMocks);


/**
 * @param {HTMLElement} screen
 * @param {String} path
 * @param {HTMLInputElement?} portElem
 * @returns {Promise<{eventId: String}>}
 */
async function getMockEventId(screen, path, portElem) {
    if (screenToEventId.has(screen)) return { eventId: screenToEventId.get(screen) };
    const mockData = { specPath: path };

    const port = portElem.value ? Number.parseInt(portElem.value) : null;
    if (Number.isNaN(port)) throw new Error("Please Enter a valid port number");
    if (port) mockData.port = port;

    const { data: eventData, error } = await makeHttpCall("/mock/start", { method: "POST", body: mockData });
    if (error) throw new Error(error);

    const fileType = screen.getAttribute('data-file-type');
    const hostPort = (() => {
        try {
            const u = new URL(eventData.url);
            const host = (u.hostname === '0.0.0.0') ? 'localhost' : u.hostname;
            return `${host}:${u.port}`;
        } catch (e) {
            const raw = (eventData.url || '').replace(/^https?:\/\//, '');
            return raw.startsWith('0.0.0.0:') ? raw.replace(/^0\.0\.0\.0/, 'localhost') : raw;
        }
    })();

    const message = fileType === 'asyncapi' ? `Kafka mock broker: ${hostPort}` : `Mock started on ${eventData.url}`;
    screenToEventId.set(screen, eventData.eventId);
    Workspace.updateInfoBox(screen, message, MOCK_INFO_SELECTOR);
    if (portElem) portElem.value = eventData.port;
    try { EphemeralStore.set(`ran-mock-${screen.id}`, true); } catch (_) { /* ignore */ }
    return { eventId: eventData.eventId };
}

async function getWorkflowMockEventId(screen, path) {
    if (screenToEventId.has(screen)) return { eventId: screenToEventId.get(screen) };
    const serverPicker = screen.querySelector(".details .mock server-picker");
    const servers = serverPicker.getServers();

    const { data, error } = await makeHttpCall("/mock/start", { method: "POST", body: { specPath: path, servers } });
    if (error) throw new Error(error);

    screenToEventId.set(screen, data.eventId);
    Workspace.updateInfoBox(screen, "Workflow mock started", MOCK_INFO_SELECTOR);
    screen.querySelector(".details .mock table").setAttribute("data-diagrams", "true");
    const drillDown = getOrPut({
        cache: screenToMockMermaidDrillDown,
        key: screen,
        defaultValue: () => new MermaidDrillDown({
            eventId: data.eventId,
            container: screen.querySelector(".mock .diagrams"),
        })
    });
    drillDown.reset(data.eventId, false);

    return { eventId: data.eventId };
}

/**
 * @param {Object} options
 * @param {String} options.eventId
 * @param {Object} options.data
 * @param {String?} options.filter
 * @returns {Promise<Object[]?>}
 */
async function getMockDetails({ eventId, data, filter }) {
    const { data: details, error } = await makeHttpCall(`/mock/${eventId}`, { method: "POST", body: data });
    if (error) {
        createAlert({ title: "Failed to get endpoint details", message: error, type: "error" });
        return null;
    }

    if (!filter || filter === "total") return details;
    return details.filter(d => d.result.toLowerCase() === filter.toLowerCase());
}

document.addEventListener("DOMContentLoaded", () => {
    // Add event listeners to all start and stop mock buttons
    document.addEventListener("click", (e) => {
        if (e.target.classList.contains("startMock")) {
            handleStartMock(e);
        } else if (e.target.classList.contains("stopMock")) {
            handleStopMock(e);
        }
    });
});

function showServerInfo(screen, metaData) {
    if (!screen) return;

    const logContent = screen.querySelector(".mockServerLogContent");
    if (!logContent) return;

    const { url, message, specPath, eventId } = metaData;
    logContent.textContent = `The mock server started on url: '${url}'`;
    logContent.scrollTop = logContent.scrollHeight;
}

async function handleStartMock(e) {
    const button = e.target;
    const screen = button.closest(".screen");
    if (!screen) return;

    const fileType = screen.getAttribute('data-file-type');
    if (fileType === "openapi") {
        throw new Error("OpenAPI file should not be started here");
    }

    const specPath = screen.id;
    if (!specPath) {
        return createAlert({
            title: "Error",
            message: "No specification file selected",
            type: "error"
        });
    }

    // Disable the start button while the request is being processed
    button.disabled = true;
    button.textContent = "Starting...";

    try {
        const { data, error } = await makeHttpCall("/mock/start", {
            method: "POST",
            body: { specPath: specPath, port: screen.querySelector("#kafkaPort")?.value || "" },
            baseUrl: BASE_URL
        });

        if (error) {
            button.disabled = false;
            button.textContent = "Start Mock Server";
            return createAlert({
                title: "Failed to start mock server",
                message: error,
                type: "error"
            });
        }

        // Show success message
        createAlert({
            title: "Mock Server Started",
            message: "Mock server started successfully",
            type: "success",
            duration: 5000
        });

        try { EphemeralStore.set(`ran-mock-${screen.id}`, true); } catch (_) { /* ignore */ }

        // Hide start button and show stop button
        button.style.display = "none";
        const stopButton = screen.querySelector(".stopMock");
        if (stopButton) {
            stopButton.disabled = false;
            stopButton.textContent = "Stop Mock Server";
            stopButton.style.display = "inline-block";
        }

        // Show log container
        const logContainer = screen.querySelector(".mockServerLog");
        if (logContainer) {
            logContainer.style.display = "block";
        }

        // Set up polling for logs
        showServerInfo(screen, data);
        screenToEventId.set(screen, data.eventId);
    } catch (err) {
        button.disabled = false;
        button.textContent = "Start Mock Server";
        createAlert({
            title: "Error",
            message: err.message || "Failed to start mock server",
            type: "error"
        });
    }
}

async function handleStopMock(e) {
    const button = e.target;
    const screen = button.closest(".screen");
    if (!screen) return;

    const specPath = screen.id;
    if (!specPath) {
        return createAlert({
            title: "Error",
            message: "No specification file selected",
            type: "error"
        });
    }

    // Disable the stop button while the request is being processed
    button.disabled = true;
    button.textContent = "Stopping...";

    try {
        const eventId = screenToEventId.get(screen);
        const { data, error } = await makeHttpCall("/mock/stop", {
            method: "POST",
            body: { id: eventId },
            baseUrl: BASE_URL
        });

        if (error) {
            button.disabled = false;
            button.textContent = "Stop Mock Server";
            return createAlert({
                title: "Failed to stop mock server",
                message: error,
                type: "error"
            });
        }

        // Show success message
        createAlert({
            title: "Mock Server Stopped",
            message: "Mock server stopped successfully",
            type: "success",
            duration: 5000
        });

        // Hide stop button and show start button
        button.style.display = "none";
        const startButton = screen.querySelector(".startMock");
        if (startButton) {
            startButton.disabled = false;
            startButton.textContent = "Start Mock Server";
            startButton.style.display = "inline-block";
        }

        // Hide log container
        const logContainer = screen.querySelector(".mockServerLog");
        if (logContainer) {
            logContainer.style.display = "none";
        }

        // Clear log content
        const logContent = screen.querySelector(".mockServerLogContent");
        if (logContent) {
            logContent.textContent = "";
        }
    } catch (err) {
        button.disabled = false;
        button.textContent = "Stop Mock Server";
        createAlert({
            title: "Error",
            message: err.message || "Failed to stop mock server",
            type: "error"
        });
    }
}

function showPreReqErrorBar(errors, screen) {
    const mockTab = screen.querySelector('.details .mock');

    if (mockTab) {
        const alertDiv = mockTab.querySelector('#prereq-error-alert');
        const summary = mockTab.querySelector('#prereq-error-summary');
        const details = mockTab.querySelector('#prereq-error-message');

        if (!alertDiv || !summary || !details) return;
        if(!errors) {
            alertDiv.open = false;
            alertDiv.style.display = "none";
            summary.textContent = null;
            details.innerHTML = null;
            return;
        }

        alertDiv.style.display = "block";
        summary.textContent = errors.length >= 1
            ? `(${errors.length}) Error/s occurred while loading some examples`
            : "Errors occured while loading some examples"

        details.innerHTML = errors.map(e =>
            `<div style="margin-bottom:8px">${e.errorMessage.replace(/\n/g, "<br>")}</div>`
        ).join("");
    }
}

function subscribeToMockEvents(screen) {
    if (!keepSubscribingToMockEvents) return;
    const unsubscription = SseEventStreamer.subscribe({
        eventId: "MOCK",
        callbacks: {
            onError: (data) => {
                const errors = JSON.parse(JSON.stringify(data));
                if (!errors || !errors.length || errors.length == 0) {
                    showPreReqErrorBar(null, screen);
                } else {
                    showPreReqErrorBar(errors, screen);
                }
                if (keepSubscribingToMockEvents) {
                    setTimeout(() => {
                        subscribeToMockEvents(screen);
                    }, 2000); 
                }
            }
        }
    });
    mockEventSubscriptions.push(unsubscription);
}

// Expose a global hook so proc.js can notify when a mock process stops (manual stop or harvest)
function onMockProcessStopped({ eventId, specPath }) {
    try {
        const screens = document.querySelectorAll(`.screen#${CSS.escape(specPath)}`);
        screens.forEach(screen => {
            const stored = screenToEventId.get(screen);
            if(stored && stored === eventId) {
                screenToEventId.delete(screen);
                const runBtn = screen.querySelector('.details .mock button.run');
                if(runBtn) runBtn.setAttribute('data-running', 'false');
                Workspace.updateInfoBox(screen, null, MOCK_INFO_SELECTOR);
                showPreReqErrorBar(null, screen);
            }
        });
    } catch(e) { /* ignore */ }
};
