/**
 * @param {Object} options
 * @param {HTMLElement} options.screen
 * @param {HTMLElement} options.tab
 * @param {Boolean} options.withSelection
 * @param {Map<HTMLElement, TableMaker>} options.tableStore
 * @param {Map<HTMLElement, TableFilter>} options.filterStore
 * @param {Set<String>} options.groupBy=new Set(["path", "method", "response"])
 */
async function init({ screen, tab, withSelection, tableStore, filterStore, groupBy = new Set(["path", "method", "response"]) }) {
    if (tableStore.has(screen)) return;
    enableActuator(screen, false);
    const countsMap = getCountsMap(tab);
    const table = tab.querySelector("table");
    if (!table) throw new Error("Table not found");

    const tableMaker = getOrPut({
        cache: tableStore,
        key: screen,
        defaultValue: () => new TableMaker({
            target: table,
            groupBy: groupBy,
            onDuplicate: ({ existingRow, data }) => duplicateHandler({ existingRow, data, countsMap }),
            onAddingRow: ({ row, data }) => onNewRow({ row, tab, data, countsMap }),
            onExtract: extractCellValue,
            creator: rowCreator,
            addSelection: withSelection
        })
    });

    const tableFilter = getOrPut({
        cache: filterStore,
        key: screen,
        defaultValue: () => new TableFilter({
            countsList: tab.querySelector(".counts"),
            countsMap: countsMap,
            tableMaker: tableMaker,
            updateGroupCoverage: ({ group, coverage }) => handleCoverageHitEvent({ tableMaker, group, coverage })
        })
    });

    await resetTableWithApi({ screen, countsMap, tableMaker, tableFilter, target: tab.className });
}

/**
 * @param {Object} options
 * @param {HTMLElement} options.screen
 * @param {Map<string, HTMLElement>} options.countsMap
 * @param {TableMaker} options.tableMaker
 * @param {TableFilter} options.tableFilter
 * @param {String} options.target
 */
async function resetTableWithApi({ screen, countsMap, tableMaker, tableFilter, target }) {
    const { data, error } = await makeHttpCall("/specifications/endpoints", { method: "GET", queryParams: { specification: screen.id, target } });
    if (error) return createAlert({ title: `Failed to get endpoints for ${screen.id}`, message: error, type: "error" });

    enableActuator(screen, false);
    const testTab = screen.querySelector('.details .test');
    if (testTab) {
        const errorAlert = testTab.querySelector('#prerequisite-error-alert');
        if (errorAlert) {
            errorAlert.style.display = 'none';
        }
    }

    tableMaker.reset();
    tableFilter.reset(tableMaker);
    tableMaker.target.classList.remove('hidden');
    setStyles({ table: tableMaker.target, id: target, countsMap, groupCount: tableMaker.groupBy.size });

    for (const endpoint of data.endpoints) {
        const data = { coverage: 0, ...endpoint, remark: undefined, result: undefined, _result: "notcovered", __skipRowResult: true }
        tableFilter.recordAndAddRow(data);
    }

    const testBaseURLInput = screen.querySelector("#testBaseUrl");
    if (testBaseURLInput && testBaseURLInput.value === "" && data.baseURL) {
        testBaseURLInput.value = data.baseURL;
    }
}

/**
 * @param {Object} options
 * @param {HTMLElement} options.screen
 * @param {HTMLElement} options.tab
 * @param {Map<HTMLElement, TableMaker>} options.tableStore
 * @param {Map<HTMLElement, DrillDown>} options.drillDownStore
 * @param {Map<HTMLElement, TableFilter>} options.filterStore
 * @param {Boolean} [options.needStart=true]
 * @param {function({selected: Object[]}): Promise<{eventId: String}>} options.getEventId
 * @param {function({eventId: String, data: Object, filter: String?}): Promise<Object[]?>} options.onDrillDown
 * @param {Set<String>} options.groupBy=new Set(["path", "method", "response"])
 * @param {function(String): void} [options.onEnd]
 */
async function run({ screen, tab, tableStore, drillDownStore, filterStore, needStart = true, getEventId, onDrillDown, onEnd = () => {}, groupBy = new Set(["path", "method", "response"]) }) {
    const countsMap = getCountsMap(tab);
    const tableMaker = tableStore.get(screen);
    const tableFilter = filterStore.get(screen);

    if (!tableMaker) throw new Error("TableMaker not found");
    if (!tableFilter) throw new Error("TableFilter not found");

    const selected = tableMaker.getSelections();
    await resetTableWithApi({ screen, countsMap, tableMaker, tableFilter, target: tab.className });
    tableMaker.disableSelectionFeature();

    const mainDiv = tableMaker.target.parentElement;
    if (!mainDiv) throw new Error("MainDiv not found");

    let eventId;
    try {
        ({ eventId } = await getEventId({ selected }));
    } catch (e) {
        createAlert({ title: `Failed to start ${tab.className}`, message: e.message, type: "error" });
        toggleStart(tab, true);
        return;
    }

    const drillDown = getOrPut({
        cache: drillDownStore,
        key: screen,
        defaultValue: () => new DrillDown({
            mainDiv,
            table: tableMaker.target,
            eventId,
            groupColumns: groupBy,
            onDrillDown: ({ eventId, data }) => onDrillDown({ data, eventId, filter: tableFilter.getCurrentFilter() }),
            onExtract: extractCellValue
        })
    });
    drillDown.updateEventId(eventId);

    const unsubscribe = SseEventStreamer.subscribe({
        eventId: eventId,
        callbacks: {
            onData: (data) => onDataHandler({ screen, data, tableMaker, tableFilter }),
            onEnd: (data) => {
                createAlert({ title: data, type: "info", duration: 3000 });
                onComplete({ tab, tableMaker, tableFilter, countsMap });
                onEnd(eventId);
            },
            onError: (data) => {
                createAlert({ title: "Something went wrong", message: data.error, type: "error" })
                onComplete({ tab, tableMaker, tableFilter, countsMap });
                onEnd(eventId);
            }
        }
    });

    if (needStart) {
        const { error } = await makeHttpCall("/task/start", { method: "POST", body: { id: eventId } });
        if (error) {
            unsubscribe();
            return createAlert({ title: `Failed to start ${tab.className}`, message: error, type: "error" });
        }
    }

    toggleStart(tab, false);
}

function onComplete({ tab, tableMaker, tableFilter, countsMap }) {
    toggleStart(tab, true);
    tableMaker.enableSelectionFeature();
    markNotCovered(tableMaker.target, countsMap);
    tableFilter.onEnd(tab.className === "test");
}

function toggleStart(tab, enable) {
    tab.querySelector("button.run").setAttribute("data-running", !enable);
}

/**
 * @param {Object} options
 * @param {Object} options.data
 * @param {TableMaker} options.tableMaker
 * @param {TableFilter} options.tableFilter
 */
function onDataHandler({ screen, data, tableMaker, tableFilter }) {
    const { type, ...payload } = data;
    switch (type) {
        case "ActuatorEndpointsEvent": {
            for (const endpoint of payload.missing) tableFilter.recordAndAddRow({ coverage: 0, ...endpoint, remark: "Missing In Spec", result: 0, _result: "notcovered" });
            break;
        }
        case "CoverageEvent": {
            const topLevelCoverage = tableMaker.target.parentElement?.parentElement?.querySelector('.details > * > .header > .coverage > span');
            if (topLevelCoverage) topLevelCoverage.textContent = `${payload.coverage}%`;
            break;
        }
        case "CoverageHitEvent": {
            tableFilter.addCoverage(payload.group, payload.coverage);
            break;
        }
        case "ActuatorEvent": {
            if (payload.enabled) {
                enableActuator(screen, true);
            }
            break;
        }
        case "PrerequisiteErrorEvent": {
            const testTab = screen.querySelector('.details .test');
            if (testTab) {
                const errorAlert = testTab.querySelector('#prerequisite-error-alert');
                const errorMessage = testTab.querySelector('#prerequisite-error-message');
                if (errorAlert && errorMessage) {
                    errorMessage.innerHTML = payload.message.replace(/\r\n\r\n/g, '\n').replace(/\n/g, '<br>');;
                    errorAlert.style.display = 'flex';
                }
            }
            break;
        }
        case "EndpointEvent": break;
        case "ArazzoMockDiagramEvent": break;
        default: tableFilter.recordAndAddRow(payload);
    }
}

/**
 * @param {HTMLElement} tab
 * @returns {Map<string, HTMLElement>}
 */
function getCountsMap(tab) {
    const countsToElem = new Map();
    const counts = tab.querySelectorAll(".header > .counts > *");
    for (const count of counts) {
        const key = count.getAttribute("data-type");
        countsToElem.set(key, count);
    }
    return countsToElem;
}

/**
 * @param {Object} options
 * @param {String} options.key
 * @param {any} options.value
 * @param {Object} options.data
 * @returns {HTMLTableCellElement}
 */
function rowCreator({ key, value, data }) {
    const td = document.createElement("td");
    td.setAttribute('data-key', key);

    if (key === "workflow") {
        td.setAttribute('data-value', value);

        const flowChartBtn = document.createElement("button");
        flowChartBtn.classList.add("flow-chart-btn");
        flowChartBtn.classList.add("hidden");
        flowChartBtn.textContent = "Flow Chart";

        const textSpan = document.createElement("span");
        textSpan.textContent = value;

        td.appendChild(textSpan);
        td.appendChild(flowChartBtn);
        return td;
    }

    if (key !== "result") {
        td.setAttribute('data-value', value);
        td.textContent = value;
        return td;
    }

    const template = document.getElementById("count-fragment");
    if (!template) throw new Error("Template not found for result");
    const nodeFromTemplate = template.content.cloneNode(true);
    td.appendChild(nodeFromTemplate);

    if (!data._result || data.__skipRowResult) return td;
    increaseCellCount({ cell: td, result: data._result.toLowerCase(), by: data.result })
    return td;
}

/**
 * @param {Object} options
 * @param {HTMLTableRowElement} options.existingRow
 * @param {Object} options.data
 * @param {Map<string, HTMLElement>} options.countsMap
 * @returns {boolean}
 */
function duplicateHandler({ existingRow, data, countsMap }) {
    const remarkTdElem = existingRow.querySelector('td[data-key="remark"]');
    remarkTdElem?.setAttribute('data-value', data.remark);
    if (remarkTdElem && remarkTdElem.textContent.toLowerCase() != "covered") remarkTdElem.textContent = data.remark;

    const coverageTdElem = existingRow.querySelector('td[data-key="coverage"]');
    coverageTdElem?.setAttribute('data-value', data.coverage);
    if (coverageTdElem) coverageTdElem.textContent = `${data.coverage}%`;

    if (!data._result || data.__skipRowResult) return false;
    const resultTdElem = existingRow.querySelector('td[data-key="result"]');
    if (!resultTdElem) throw new Error("result td not found");
    increaseCellCount({ cell: resultTdElem, result: data._result, by: data.result });

    if (data.__skipCount) return false;
    updateCounts({ result: data._result, countsMap, by: data.result });
    return false;
}

/**
 * @param {Object} options
 * @param {HTMLTableRowElement} options.row
 * @param {HTMLElement} options.tab
 * @param {Map<string, HTMLElement>} options.countsMap
 * @param {Object} options.data
 * @returns {void}
 */
function onNewRow({ row, tab, countsMap, data }) {
    if (data._requestContentType) {
        const responseCell = row.querySelector('td[data-key="response"]');
        responseCell?.classList.add("response-cell")
        const span = document.createElement('span');
        span.textContent = data._requestContentType;
        responseCell?.appendChild(span);
    }

    if (tab.className === "test" && data.remark?.toLowerCase() === "missing in spec") {
        const remarkTd = row.querySelector('td[data-key="remark"]');
        if (!remarkTd) throw new Error("remark td not found");
        const excludeBtn = document.createElement('button');
        excludeBtn.textContent = 'Exclude';
        excludeBtn.classList.add('exclude-btn');
        remarkTd.appendChild(excludeBtn);
    }

    if (data.__skipCount || !data.result) return;
    updateCounts({ result: data._result, countsMap, by: data.result });
}

function extractCellValue({ key, cell }) {
    const values = {}
    values[key] = cell.getAttribute('data-value');
    if (key === 'response' && cell.querySelector('span') !== null) values.requestContentType = cell.querySelector('span').textContent;
    return values
}

/**
 * @param {Object} options
 * @param {String} options.result
 * @param {Map<string, HTMLElement>} options.countsMap
 * @param {Number} options.by
 * @returns {void}
 */
function updateCounts({ countsMap, result, by }) {
    const countSpan = countsMap.get(result.toLowerCase());
    if (!countSpan) throw new Error(`Count span not found for ${result}`);
    modifyChildSpanBy({ element: countSpan, count: by });

    const totalSpan = countsMap.get('total');
    if (!totalSpan) throw new Error("Total span not found");
    modifyChildSpanBy({ element: totalSpan, count: by });
}

/**
 * @param {Object} options
 * @param {Element} options.element
 * @param {Number} [options.count=1]
 * @param {Boolean} [options.reset=false]
 */
function modifyChildSpanBy({ element, count = 1, reset = false }) {
    const countSpan = element.querySelector('span');
    if (!countSpan) throw new Error("Count span not found");
    const newValue = reset ? 0 : Math.max(Number.parseInt(countSpan.textContent || '0') + count, 0);
    if(element.classList.contains('count')) {
        if (newValue === 0) {
            element.classList.add('disabled');
        } else {
            element.classList.remove('disabled');
        }
    }
    countSpan.textContent = newValue.toString();
    countSpan.setAttribute('data-value', newValue.toString());
}

/**
 * @template T
 * @template U
 * @param {{cache: Map<T, U>, key: T, defaultValue: () => U}} options
 * @returns {U}
 */
function getOrPut({ cache, key, defaultValue }) {
    if (cache.has(key)) {
        const value = cache.get(key);
        if (value) return value;
    }

    const value = defaultValue();
    cache.set(key, value);
    return value;
}

/**
 * @param {Object} options
 * @param {HTMLTableElement} options.table
 * @param {String} options.id
 * @param {Map<string, HTMLElement>} options.countsMap
 */
function setStyles({ table, id, countsMap, groupCount }) {
    const cleanedId = id.replace(/^[^A-Za-z]+/, '')
    table.id = cleanedId;

    const style = document.createElement('style');
    style.textContent = `
#${cleanedId} > tbody > tr > td:nth-last-child(2) {
    text-transform: capitalize;
}

#${cleanedId} > tbody > tr > td[data-key="selection"]{
    width: 0.5rem;
    &:not([disabled]) {
        pointer-events: all !important;
    }
}

#${cleanedId} > tbody > tr > td[data-key="coverage"] {
    width: 0.5rem;
}

#${cleanedId} > tbody > tr > td:nth-child(-n+${groupCount+1}) {
    background-color: rgb(var(--white));
    word-break: break-word;
    pointer-events: none;
}

#${cleanedId} > tbody > tr > td:nth-last-child(-n+${groupCount+1}) {
    cursor: pointer;
}
    `;

    document.head.appendChild(style);
    countsMap.forEach((value, _) => modifyChildSpanBy({ element: value, reset: true }));

    const topLevelCoverage = table.parentElement?.parentElement?.querySelector('.details > * > .header > .coverage');
    if (topLevelCoverage) modifyChildSpanBy({ element: topLevelCoverage, reset: true });
}

/**
 * @param {Object} options
 * @param {Element} options.cell
 * @param {String} options.result
 * @param {Number} options.by
 */
function increaseCellCount({ cell, result, by }) {
    const totalSpan = cell.querySelector('span[data-key="total"]');
    if (!totalSpan) throw new Error("Total span not found");
    const newTotal = Number.parseInt(totalSpan.getAttribute('data-value') || '0') + by;
    totalSpan.setAttribute('data-value', newTotal.toString());

    const countSpan = cell.querySelector(`span[data-key="${result.toLowerCase()}"]`);
    if (!countSpan) throw new Error(`Count span not found for ${result}`);
    const newCount = Number.parseInt(countSpan.getAttribute('data-value') || '0') + by;
    countSpan.setAttribute('data-value', newCount.toString());

    const nonTotalSpans = Array.from(cell.querySelectorAll('span.count:not([data-key="total"])'));
    const nonZeroCount = nonTotalSpans.filter(s => Number.parseInt(s.getAttribute('data-value') || '0') > 0).length;
    totalSpan.toggleAttribute('hidden', nonZeroCount === 1);
}

/**
 * @param {Object} options
 * @param {TableMaker} options.tableMaker
 * @param {string} options.group
 * @param {number|string} options.coverage
 */
function handleCoverageHitEvent({ tableMaker, group, coverage }) {
    const rows = Array.from(tableMaker.tbody?.querySelectorAll("tr") || []);
    const mainGroupKey = tableMaker.getMainGroupKey();
    const [first, ...rest] = rows.filter((tr) => {
        return tr.querySelector(`td[data-key=${mainGroupKey}]`)?.getAttribute("data-value") === group;
    });

    if (first === undefined) return;
    for (const row of [first, ...rest]) {
        const coverageTd = row.querySelector('td[data-key="coverage"]');
        if (!coverageTd) throw new Error("Coverage td not found");
        coverageTd.classList.add('hidden');
        coverageTd.removeAttribute('data-main');
        coverageTd.textContent = typeof coverage === "number" ? `${coverage}%` : coverage;
    }

    const firstCoverageTd = first.querySelector('td[data-key="coverage"]');
    if (!firstCoverageTd) throw new Error("First coverage td not found");
    firstCoverageTd.classList.remove('hidden');
    firstCoverageTd.setAttribute('data-main', 'true');
    firstCoverageTd.setAttribute('rowspan', (rest.length + 1).toString());
}

function markNotCovered(table, countsMap) {
    const rows = Array.from(table.querySelectorAll('tbody > tr')).filter(tr => tr.querySelector('td[data-key="remark"]').getAttribute('data-value') === "undefined");

    for (const row of rows) {
        const remarkTd = row.querySelector('td[data-key="remark"]');
        if (!remarkTd) throw new Error("remark td not found");
        if (remarkTd.getAttribute("data-value") === "undefined") {
            remarkTd.setAttribute('data-value', 'notcovered');
            remarkTd.textContent = 'Not Covered';
        }

        const resultTdElem = row.querySelector('td[data-key="result"]');
        if (!resultTdElem) throw new Error("result td not found");
        increaseCellCount({ cell: resultTdElem, result: "NotCovered", by: 1 });
    }

    updateCounts({ result: "notcovered", countsMap, by: rows.length });
}

function enableActuator(screen, enable) {
    if (!screen.querySelector("#actuator-alert")) return;
    if (enable) {
        screen.querySelector("#actuator-alert").style.display = "block";
    } else {
        screen.querySelector("#actuator-alert").style.display = "none";
    }
}
