class TableMaker {

    /**
     * @param {Object} options
     * @param {HTMLTableElement} options.target
     * @param {Set<string>} options.groupBy
     * @param {Boolean} [options.addSelection="false"]
     * @param {function({key: string, value: any, data: Object}): HTMLTableCellElement} [options.creator]
     * @param {function({existingRow: HTMLTableRowElement, data: Object}): Boolean} [options.onDuplicate]
     * @param {function({row: HTMLTableRowElement, data: Object}): void} [options.onAddingRow]
     * @param {function({key: String, cell: HTMLTableCellElement}): Object} [options.onExtract]
     */
    constructor({ target, groupBy, addSelection = false, creator, onDuplicate, onAddingRow, onExtract }) {
        this.target = target
        this.groupBy = groupBy;
        this.onAddingRow = onAddingRow
        this.addSelection = addSelection;
        this.newRowSelection = addSelection;
        this.onDuplicate = onDuplicate || (() => true);
        this.onExtract = onExtract || (({ cell }) => cell.getAttribute('data-value'));
        this.creator = creator || (({ value }) => {
            const td = document.createElement("td");
            td.textContent = value;
            return td;
        });


        this.columns = Array.from(target?.querySelectorAll("th") || []).map(td => td.innerText.trim());
        this.mainGroups = new Map();
        if (target.querySelector("tbody")) {
            this.tbody = target.querySelector("tbody");
        } else {
            this.tbody = document.createElement("tbody");
            target.appendChild(this.tbody);
        }

        const hasSelectionTh = target.querySelector("thead > tr > th:first-child")?.getAttribute('data-key') === 'selection';
        if (this.addSelection && !hasSelectionTh) {
            const thR = target.querySelector("thead > tr");
            const wrapper = document.createElement("th");
            const input = document.createElement("input");
            input.type = "checkbox";
            input.addEventListener("change", () => this._flipSelections(input.checked));
            wrapper.appendChild(input);
            wrapper.setAttribute('data-key', 'selection');
            thR?.insertBefore(wrapper, thR.firstChild);
        }
    }

    reset() {
        this.mainGroups.clear();
        if (this.addSelection) this.target.querySelector("thead > tr > th:first-child > input").checked = false;
        this.target.querySelector("tbody")?.replaceChildren();
    }

    _flipSelections(enable) {
        const checkboxes = Array.from(this.target.querySelectorAll("input[type=checkbox]"));
        for (const checkbox of checkboxes) checkbox.checked = enable;
    }

    /**
     * @param {HTMLElement} target
     */
    _addSelection(target) {
        if (!this.addSelection) return;
        const td = document.createElement("td");
        const input = document.createElement("input");
        input.type = "checkbox";
        if (!this.newRowSelection) td.setAttribute('disabled', "true");
        td.setAttribute('data-key', 'selection');
        td.appendChild(input);
        target.insertBefore(td, target.firstChild);
    }

    /**
     * @param {HTMLTableRowElement} row
     */
    _extractRowValues(row) {
        let values = {};
        for (const column of this.groupBy) {
            const cell = row.querySelector(`td[data-key="${column}"]`);
            values = { ...values, ...this.onExtract({ key: column, cell: cell }) };
        }
        return values;
    }

    getSelections() {
        if (!this.addSelection) return [];
        const rows = Array.from(this.tbody.querySelectorAll("tr"));
        const selectedRows = rows.filter(row => row?.querySelector("input")?.checked === true);
        return selectedRows.map(row => this._extractRowValues(row));
    }

    enableSelectionFeature() {
        if (!this.addSelection) return;
        this.newRowSelection = true;
        const thR = this.target.querySelector("thead > tr > th:first-child")?.removeAttribute('disabled');
        for (const row of this.tbody?.querySelectorAll("tr") || []) {
            row.querySelector("td:first-child")?.removeAttribute('disabled');
        }
    }

    disableSelectionFeature() {
        if (!this.addSelection) return;
        this.newRowSelection = false;
        this.target.querySelector("thead > tr > th:first-child")?.setAttribute("disabled", "true");
        for (const row of this.tbody?.querySelectorAll("tr") || []) {
            row.querySelector("td:first-child")?.setAttribute("disabled", "true");
        }
    }

    applyOnEach(fn) {
        for (const row of this.tbody?.querySelectorAll("tr") || []) {
            fn(row);
        }
    }

    /**
     * @param {Object<string, any>} data
    */
    addRow(data, inReqAnimFrame=true) {
        const shouldProceed = this._handleIfDuplicate(data);
        if (!shouldProceed) return;

        const insertBeforeOptions = [];
        const tr = document.createElement("tr");
        this._addSelection(tr);
        let keySoFar = "";

        for (const [key, value] of Object.entries(data)) {
            if (key.startsWith("__")) continue;
            if (key.startsWith("_")) {
                tr.setAttribute(`data-${key}`, value);
                continue;
            }

            const td = this.creator({ key, value, data });
            td.setAttribute("data-key", key);
            td.setAttribute("data-value", value);

            if (this.groupBy.has(key)) {
                keySoFar += `-${value}`;
                this._updateElseAdd({ keySoFar, currentTableData: td, currentTableRow: tr, insertBeforeOptions });
            }

            tr.appendChild(td);
        }

        this.onAddingRow?.({ row: tr, data });
        const finalInsertBefore = insertBeforeOptions.filter(t => t).pop();
        this._insertRow(inReqAnimFrame, tr, finalInsertBefore);
    }

    _insertRow(inReqAnimFrame, tr, insertBefore) {
        if (inReqAnimFrame) requestAnimationFrame(() => this.tbody?.insertBefore(tr, insertBefore?.nextSibling));
        else this.tbody?.insertBefore(tr, insertBefore?.nextSibling);
    }

    /**
     * @param {Object<string, any>} data
     * @returns {boolean}
    */
    _handleIfDuplicate(data) {
        const finalKey = this.getKeyFor(data);
        if (this.mainGroups.has(finalKey)) {
            return this.onDuplicate({ existingRow: this.mainGroups.get(finalKey).lastRow, data });
        }

        return true;
    }

    getKeyFor(data) {
        return `-${Object.entries(data).filter(([key]) => this.groupBy.has(key)).map(([_, value]) => value).join("-")}`
    }

    /**
     * @param {Object<string, any>} data
     * @returns {Object<string, string>}
     */
    getMainGroupEntry(data) {
        const mainKey = Array.from(this.groupBy)[0];
        const value = data[mainKey];
        return { key: mainKey, value };
    }

    /**
     * @returns {string}
     */
    getMainGroupKey() {
        return Array.from(this.groupBy)[0];
    }

    /**
     * @param {Object} params
     * @param {string} params.keySoFar
     * @param {HTMLElement} params.currentTableData
     * @param {HTMLElement} params.currentTableRow
     * @param {Array<HTMLElement>} params.insertBeforeOptions
     */
    _updateElseAdd({ keySoFar, currentTableData, currentTableRow, insertBeforeOptions }) {
        const existing = this.mainGroups.get(keySoFar);
        if (existing) {
            existing.td.setAttribute("rowspan", ++existing.count);
            currentTableData.classList.add("hidden");
            insertBeforeOptions.push(existing.lastRow);
            existing.lastRow = currentTableRow;
        } else {
            currentTableData.setAttribute("data-main", "true");
            this.mainGroups.set(keySoFar, { count: 1, td: currentTableData, lastRow: currentTableRow });
        }
    }

    _modifyGroup({ data, shared }) {
        let keySoFar = "";
        for (const [key, value] of Object.entries(data).filter(([key]) => this.groupBy.has(key))) {
            keySoFar += `-${value}`;
            const existing = this.mainGroups.get(keySoFar);
            if (!existing) throw new Error(`Existing not found for ${keySoFar}`);

            const newRowSpan = Number.parseInt(existing.td.getAttribute("rowspan") || "0") - 1;
            existing.td.setAttribute("rowspan", newRowSpan);
            existing.count = newRowSpan;
            existing.lastRow = existing.lastRow.previousElementSibling;

            if (shared.has(key)) {
                const selector = existing.td.parentElement.querySelector(`td[data-key="${shared.get(key)}"]`);
                selector.setAttribute("rowspan", newRowSpan);
            }

            if (newRowSpan < 0) this.mainGroups.delete(keySoFar);
        }
    }
}
