const leftSideBar = document.getElementById("left-sidebar");
const typeToIcons = new Map([
    // Specifications
    ["openapi", "icon icon-openapi"],
    ["asyncapi", "icon icon-asyncapi"],
    ["arazzo", "icon icon-arazzo"],
    // Files
    ["json", "icon icon-json"],
    ["yaml", "bi bi-filetype-yml"],
    // Operations
    ["operation", "bi bi-list-nested"],
    ["get", "icon icon-get"],
    ["post", "icon icon-post"],
    ["put", "icon icon-put"],
    ["patch", "icon icon-patch"],
    ["delete", "icon icon-delete"],
    ["head", "icon icon-head"],
    ["send", "bi bi-send-fill"],
    ["receive", "bi bi-send-fill bi-receive"]
]);
const specTypes = new Set(["openapi", "asyncapi", "arazzo"]);
const nonOpaqueTypes = [...typeToIcons.keys()].filter(type => type !== "json" && type !== "yaml");

let _fileTree;
let _treeInitialized = false;

function destroyExistingTree(treeContainer) {
    if (!_fileTree) return;
    try {
        // Wunderbaum does not document a destroy in this snippet; attempt to clear DOM and drop reference
        treeContainer.innerHTML = "";
    } catch (e) {
        console.warn("Failed to destroy existing tree", e);
    } finally {
        _fileTree = null;
        _treeInitialized = false;
    }
}

function initializeSpecTree() {
    const treeContainer = document.getElementById("spec-tree");
    const loader = document.getElementById("spec-tree-loader");
    if (!treeContainer || !loader) return;

    // If already initialized, reset first
    if (_treeInitialized) {
        destroyExistingTree(treeContainer);
    }

    loader.style.display = 'block';
    treeContainer.style.display = 'none';

    _fileTree = new mar10.Wunderbaum({
        source: {
            url: `${BASE_URL}/specifications/tree`
        },
        element: treeContainer,
        lazyLoad: _loadSpecs.bind(this),
        activate: _navigateTo.bind(this),
        tooltip: true,
        icon: (e) => typeToIcons.get(e.node.type) || true,
        init: () => {
            loader.style.display = 'none';
            treeContainer.style.display = 'block';
            _treeInitialized = true;
        },
        receive: (e) => {
            return e.response.map((elem) => {
                const unselectable = elem.unselectable;

                if (elem.type !== "folder" && !unselectable) {
                    Workspace.addScreen(elem.title, elem.path, elem.type, elem.displayablePath);
                }
                return {
                    ...elem,
                    unselectable,
                    lazy: elem.hasMoreChildren
                };
            });
        },
        render: (e) => {
            setupDraggable(e);
            e.nodeElem.setAttribute("data-type", e.node.type);
            e.nodeElem.setAttribute("data-file-path", e.node.data.path);
            
            if (e.node.unselectable && !nonOpaqueTypes.includes(e.node.type)) {
                e.nodeElem.style.opacity = 0.5;
            }
        }
    });
}

async function _navigateTo(e) {
    if (e.event?.target?.tagName === "I") return;
    const node = e.node;
    const isSpecmaticConfig = node?.title === "specmatic.yaml";
    if (!specTypes.has(node.type) && !isSpecmaticConfig) return;
    Workspace.switchTo(node.data.path, false);
}

async function _loadSpecs(e) {
    const { data, error } = await makeHttpCall("/specifications/tree", { method: "GET", queryParams: { path: e.node.data.path } });
    if (error) throw new Error(error);
    if (data.length === 0) e.node.setStatus("noData");
    return data
}

function _sortTree(node, deep) {
    node.sortChildren((a, b) => {
        if (a.type === "folder" && b.type !== "folder") return -1;
        if (a.type !== "folder" && b.type === "folder") return 1;
        return a.title.localeCompare(b.title);
    }, deep);
}

function addNode({ title, path, type, unselectable, highlight, displayablePath }) {
    const parentPath = getParentPath(path);
    const parentNode = _fileTree.findFirst((node) => node.data.path === parentPath) || _fileTree.root;

    let node = parentNode.findFirst((node) => node.data.path === path);
    if (!node) {
        node = parentNode.addNode({ title, path, type, lazy: type === "folder" || type === "openapi", unselectable });
        if (type !== "folder" && !unselectable) Workspace.addScreen(title, path, type, displayablePath);
        _sortTree(parentNode, false);
    }

    if (!highlight) return;
    highlightAndSwitchTo({ node });
}

function waitForPath({ path, retryMs = 250, maxWait = 2000 }) {
    return new Promise(resolve => {
        const start = Date.now();
        const check = () => {
            const switchNode = _fileTree.findFirst(n => n.data.path === path);
            if (switchNode) return resolve(switchNode);
            if (Date.now() - start > maxWait) return resolve(null);
            setTimeout(check, retryMs);
        };
        check();
    });
}

function highlightAndSwitchTo({ node, path }) {
    const switchNode = node || _fileTree.findFirst((node) => node.data.path === path);
    if (!switchNode) {
        console.warn("Node not found for path:", path);
        return;
    }

    switchNode.setClass("highlight");
    leftSideBar?.setAttribute("aria-expanded", "true");
    setTimeout(() => {
        switchNode.setClass("highlight", false);
        leftSideBar?.setAttribute("aria-expanded", "false");
        switchNode.setActive(true);
    }, 2000);
}

async function reloadDirectoryNode(dirPath) {
    const node = _fileTree.findFirst(n => n.data.path === dirPath);
    if (!node) {
        console.warn("Node not found for path:", dirPath);
        return;
    }

    if (node.type !== "folder" && node.type !== "openapi") {
        console.warn("Node cannot be reloaded:", dirPath);
        return;
    }

    const nodeExpandedStatus = node.expanded;
    _fileTree.runWithDeferredUpdate(() => {
        node.resetLazy();
        node.setExpanded(nodeExpandedStatus);
    });
}

async function deleteNode(path) {
    const node = _fileTree.findFirst(n => n.data.path === path);
    if (!node) {
        console.warn("Node not found for path:", path);
        return;
    }

    if (node === _fileTree.root) {
        console.warn("Root node cannot be deleted:", path);
        return;
    }

    _fileTree.runWithDeferredUpdate(() => {
        node.remove();
    });
}

function setupDraggable(elem) {
    if (elem.node.parent.type !== "operation") return;
    elem.nodeElem.draggable = true;
    elem.nodeElem.addEventListener('dragstart', (e) => {
        e.dataTransfer.setData("methodOrAction", elem.node.type.toUpperCase());
        e.dataTransfer.setData("specificationFile", elem.node.data.path);
        e.dataTransfer.setData("specType", elem.node.parent.parent.type);
        e.dataTransfer.setData("name", elem.node.title);
        if (elem.node.data.data.operationId) e.dataTransfer.setData("operationId", elem.node.data.data.operationId);
        if (elem.node.data.data.operationPath) e.dataTransfer.setData("operationPath", elem.node.data.data.operationPath);
        if (elem.node.data.data.channelPath) e.dataTransfer.setData("channelPath", elem.node.data.data.channelPath);
    });
}

document.addEventListener("DOMContentLoaded", () => {
    initializeSpecTree();
});

// Expose for reset handling
window.initializeSpecTree = initializeSpecTree;
