class DrillDown {
    /**
     * @param {Object} options
     * @param {HTMLDivElement} options.mainDiv
     * @param {HTMLElement} options.mainDiv
     * @param {Set<String>} options.groupColumns
     * @param {String} options.eventId
     * @param {function({key: String, cell: HTMLTableCellElement}): Object} options.onExtract
     * @param {function({ eventId: String, data: Object }): Promise<Object[]?>} options.onDrillDown
     */
    constructor({ mainDiv, table, eventId, groupColumns, onExtract, onDrillDown }) {
        this.mainDiv = mainDiv;
        this.table = table;
        this.groupBy = groupColumns;
        this.eventId = eventId;
        this.onDrillDown = onDrillDown;
        this.onExtract = onExtract || (({ key, cell }) => { return { key: cell.getAttribute('data-value') }});
        this.scrollPos = { top: 0, left: 0 };

        // Context detection for conditional rendering (e.g., async mock drilldown)
        try {
            const screenEl = this.mainDiv.closest('.screen');
            this._fileType = screenEl?.getAttribute('data-file-type') || '';
            this._isMockTab = !!this.mainDiv.closest('.mock');
        } catch (_) {
            this._fileType = '';
            this._isMockTab = false;
        }

        mainDiv.querySelector('div.details')?.remove();
        this.detailsDiv = document.createElement('div');
        this.detailsDiv.classList.add('details');
        mainDiv.appendChild(this.detailsDiv);
        this._setupClasses();
        this._addDetailsDiv();
        this._addListeners();
    }

    updateEventId(eventId) {
        this.eventId = eventId;
    }

    _setupClasses() {
        this.mainDiv.classList.add('drill-down');
    }

    _addListeners() {
        this.table.addEventListener('click', this._handleClick.bind(this));
        this.detailsDiv.querySelector('button')?.addEventListener('click', () => {
            this.detailsDiv.querySelector('.body')?.replaceChildren();
            this.mainDiv.setAttribute('data-mode', 'table');
            this.mainDiv.scroll({ top: this.scrollPos.top, left: this.scrollPos.left, behavior: 'smooth' });
        });
    }

    _addDetailsDiv() {
        const headerElem = document.createElement('div');
        headerElem.classList.add('header');

        const backBtn = this._backBtn();
        headerElem.appendChild(backBtn);

        const listElem = document.createElement('ul');
        listElem.classList.add('info-list');
        headerElem.appendChild(listElem);

        const bodyElem = document.createElement('div');
        bodyElem.classList.add('body');

        this.detailsDiv.appendChild(headerElem);
        this.detailsDiv.appendChild(bodyElem);
    }

    _backBtn() {
        const spanElem = document.createElement('span');
        spanElem.textContent = '←';
        const paraElem = document.createElement('p');
        paraElem.textContent = 'Go Back';
        const buttonElem = document.createElement('button');
        buttonElem.appendChild(spanElem);
        buttonElem.appendChild(paraElem);
        return buttonElem;
    }

    /**
     * @param {MouseEvent} e
     */
    async _handleClick(e) {
        const target = e.target;
        if (target?.tagName === 'BUTTON') return;
        const nearestInput = target?.closest("[data-key=selection]");
        if (nearestInput) return;

        const nearestTableRow = target?.closest("tr");
        if (!nearestTableRow) return;

        this.detailsDiv.querySelector('.body')?.replaceChildren();
        const rowValues = this._extractRowValues(nearestTableRow);
        const data = await this.onDrillDown({ eventId: this.eventId, data: rowValues });
        if (!data) return createAlert({ title: "No data found", type: "info", duration: 1000 });

        this.mainDiv.setAttribute('data-mode', 'details');
        this._addGroupDetails(rowValues);

        for (const dataItem of data) {
            const dropDownDiv = this._createDropDownDiv(dataItem);
            this.detailsDiv.querySelector('.body')?.appendChild(dropDownDiv);
        }

        this.scrollPos = { top: this.mainDiv.scrollTop, left: this.mainDiv.scrollLeft };
        this.mainDiv.scroll({ top: 0, left: 0, behavior: 'smooth' });
    }

    _addGroupDetails(data) {
        const detailsOl = this.detailsDiv.querySelector('.info-list');
        detailsOl?.replaceChildren();

        for (const [key, value] of Object.entries(data)) {
            const liElem = document.createElement('li');

            const titleElem = document.createElement('span');
            titleElem.textContent = `${key}: `;
            liElem.appendChild(titleElem);

            const textElem = document.createElement('p');
            textElem.textContent = value;
            liElem.appendChild(textElem);

            detailsOl?.appendChild(liElem);
        }
    }

    /**
     * @param {HTMLTableRowElement} row
     * @returns {Object}
     */
    _extractRowValues(row) {
        let values = {};
        for (const column of this.groupBy) {
            const cell = row.querySelector(`td[data-key="${column}"]`);
            values = {...values, ...this.onExtract({ key: column, cell })};
        }
        return values;
    }

    _createDropDownDiv(data) {
        const dropDownDiv = document.createElement('div');
        dropDownDiv.classList.add('drop-down');
        dropDownDiv.setAttribute('data-expand', "false");

        const header = this._getHeaders(data);
        header.addEventListener('click', () => dropDownDiv.setAttribute('data-expand', dropDownDiv.getAttribute('data-expand') === "true" ? "false" : "true"));
        dropDownDiv.appendChild(header);

        const details = this._createDetailsDiv(data);
        dropDownDiv.appendChild(details);

        return dropDownDiv;
    }

    _getHeaders(data) {
        const headerElem = document.createElement('div');
        headerElem.classList.add('header');
        
        const chevronImg = document.createElement('img');
        chevronImg.src = 'static/dropdown-chevron.svg';  
        chevronImg.alt = 'Chevron';
        chevronImg.classList.add('dropdown-chevron');

        const titleElem = document.createElement('p');
        titleElem.textContent = data.name;
        headerElem.appendChild(chevronImg);
        headerElem.appendChild(titleElem);

        headerElem.addEventListener('click', () => {
            chevronImg.classList.toggle('rotated');
        });

        const wrapperDiv = document.createElement('div');
        const timeElem = document.createElement('p');
        timeElem.textContent = data.time;
        wrapperDiv.appendChild(timeElem);
        const resultElem = document.createElement('p');
        resultElem.textContent = data.result;
        this._addAttributeToResult(resultElem, data.result);
        wrapperDiv.appendChild(resultElem);
        headerElem.appendChild(wrapperDiv);

        return headerElem;
    }

    _createDetailsDiv(data) {
        const detailsDiv = document.createElement('div');
        detailsDiv.classList.add('details');
        detailsDiv.appendChild(this._createDetailsSection(data.details));
        const isAsyncMock = (this._fileType === 'asyncapi') && this._isMockTab;

        if (isAsyncMock) {
            const messages = this._splitBySeparator(data.message || data.request || data.response);
            messages.forEach(msg => detailsDiv.appendChild(this._createBlock('Message:', 'message', msg)));
        } else {
            const requests = this._splitBySeparator(data.request);
            const responses = this._splitBySeparator(data.response);
            const maxLen = Math.max(requests.length, responses.length);
            for (let i = 0; i < maxLen; i++) {
                const pairWrapper = document.createElement('div');
                pairWrapper.classList.add('pair');
                const requestContent = requests[i] !== undefined ? requests[i] : 'No request';
                const responseContent = responses[i] !== undefined ? responses[i] : 'No response';
                pairWrapper.appendChild(this._createBlock('Request:', 'request', requestContent));
                pairWrapper.appendChild(this._createBlock('Response:', 'response', responseContent));
                detailsDiv.appendChild(pairWrapper);
            }
        }

        return detailsDiv;
    }

    _createDetailsSection(details) {
        const wrapper = document.createElement('div');
        const para = document.createElement('p');
        para.textContent = "Details:";
        wrapper.appendChild(para);

        const pre = document.createElement('pre');
        pre.classList.add('details');
        pre.textContent = details;
        wrapper.appendChild(pre);

        return wrapper;
    }

    _createBlock(label, cssClass, content) {
        const wrapper = document.createElement('div');
        const para = document.createElement('p');
        para.textContent = label;
        wrapper.appendChild(para);

        const pre = document.createElement('pre');
        pre.classList.add(cssClass);
        pre.textContent = content || '';
        wrapper.appendChild(pre);

        return wrapper;
    }

    _splitBySeparator(text) {
        if (!text) return [];
        return text.split('---END BLOCK---').map(s => s.trim()).filter(Boolean);
    }

    _addAttributeToResult(resultElem, result) {
        resultElem.classList.add("pill")
        switch (result.toLocaleLowerCase()) {
            case "success":
                resultElem.classList.add('green');
                break;
            case "notcovered":
                resultElem.classList.add('yellow');
                break;
            default:
                resultElem.classList.add('red');
                break;
        }
    }
}
