const alertContainer = document.querySelector("#alert-container");
const specmaticStudioTitle = document.querySelector("#specmaticStudioTitle")
const importButtons = document.querySelectorAll(".import-btn");
const proxyButtons = document.querySelectorAll(".proxy-btn");
const workflowCtaButton = document.querySelector("#workflowCta");
const exportSpecmaticBtn = document.querySelector("#exportSpecmaticBtn");
const CONFIG_ENABLED = (document.body?.getAttribute('data-config-enabled') || 'true') === 'true';

const BASE_URL_SOURCE_PRIORITY = {
  template: 0,
  spec: 1,
  config: 2,
  user: 3
};

function ensureBaseUrlInputState(input) {
  if (!input || input.dataset.baseUrlInitialized === 'true') return;
  input.dataset.baseUrlInitialized = 'true';
  if (!input.dataset.baseUrlSource) input.dataset.baseUrlSource = 'template';
  input.dataset.programmaticValue = input.value ?? '';
  if (!input.dataset.userEdited) input.dataset.userEdited = 'false';
}

function setBaseUrlInputValue(input, newValue, source) {
  if (!input) return false;
  ensureBaseUrlInputState(input);

  const effectiveSource = source || 'template';

  const userOverride = detectUserOverride(input);
  if (userOverride && effectiveSource !== 'user') return false;

  const currentSource = input.dataset.baseUrlSource || 'template';
  if (currentSource === 'user' && effectiveSource !== 'user') return false;

  const currentPriority = BASE_URL_SOURCE_PRIORITY[currentSource] ?? 0;
  const nextPriority = BASE_URL_SOURCE_PRIORITY[effectiveSource] ?? 0;
  if (nextPriority < currentPriority) return false;

  const valueToApply = newValue ?? '';
  if (input.value !== valueToApply) input.value = valueToApply;
  input.dataset.baseUrlSource = effectiveSource;
  input.dataset.programmaticValue = valueToApply;
  input.dataset.userEdited = effectiveSource === 'user' ? 'true' : 'false';
  return true;
}

window.ensureBaseUrlInputState = ensureBaseUrlInputState;
window.setBaseUrlInputValue = setBaseUrlInputValue;
window.BASE_URL_SOURCE_PRIORITY = BASE_URL_SOURCE_PRIORITY;
window.detectUserOverride = detectUserOverride;

function detectUserOverride(input) {
  if (!input) return false;
  if (input.dataset.baseUrlSource === 'user') {
    input.dataset.userEdited = 'true';
    return true;
  }

  const programmaticValue = input.dataset.programmaticValue;
  if (programmaticValue === undefined) return false;

  if (input.value !== programmaticValue) {
    input.dataset.baseUrlSource = 'user';
    input.dataset.userEdited = 'true';
    return true;
  }
  return false;
}

specmaticStudioTitle?.addEventListener("click", async () => await Workspace.switchTo(Workspace.HOME_SCREEN_ID));
workflowCtaButton.addEventListener("click", async () => await Workspace.switchTo(Workspace.WORKFLOW_SCREEN_ID))
proxyButtons?.forEach(btn => {
  const tooltip = new bootstrap.Tooltip(btn);
  btn.addEventListener("click", async () => {
    tooltip.hide();
    btn.blur();
    await Workspace.switchTo(Workspace.PROXY_SCREEN_ID)
  });
});
importButtons?.forEach(btn => {
  const tooltip = new bootstrap.Tooltip(btn);
  btn.addEventListener("click", async () => {
    tooltip.hide();
    btn.blur();
    await uploadFiles()
  });
});

exportSpecmaticBtn?.addEventListener("click", async () => {
  try {
    const { provides, consumes } = collectSpecmaticExportData();
    if (provides.length === 0 && consumes.length === 0) {
      createAlert({ title: "Nothing to export", message: "No base URLs or ports configured in open specs.", type: "info", duration: 3000 });
      return;
    }

    const { data, error } = await makeHttpCall("/config/export", { method: "POST", body: { provides, consumes } });
    if (error) return createAlert({ title: "Export failed", message: error, type: "error" });
    const outputPath = (data && data.relativePath) ? data.relativePath : "specmatic.yaml";
    createAlert({ title: "Exported specmatic.yaml", message: `Path: ${outputPath}` , type: "success", duration: 4000 });
  } catch (e) {
    createAlert({ title: "Export failed", message: e?.message || String(e), type: "error" });
  }
});

function collectSpecmaticExportData() {
  /** @type {{specification: string, baseUrl: string, resiliencyTests?: {enable: string}}[]} */
  const provides = [];
  /** @type {{specification: string, port: number}[]} */
  const consumes = [];

  const screens = document.querySelectorAll('.screen');
  screens.forEach(screen => {
    const fileType = screen.getAttribute('data-file-type');
    const specPath = screen.id;
    if (!specPath) return;
    const ranTest = !!EphemeralStore.get(`ran-test-${specPath}`, false);
    const ranMock = !!EphemeralStore.get(`ran-mock-${specPath}`, false);

    // Provides: OpenAPI specs with a Test Base URL configured
    if (fileType === 'openapi') {
      const baseUrlInput = screen.querySelector('#testBaseUrl');
      const baseUrl = baseUrlInput?.value?.trim();
      const isGenerative = !!screen.querySelector('#generative')?.checked;
      // Only add a provides entry if tests were run, not merely mocks
      if (ranTest && baseUrl) {
        const provideEntry = { specification: specPath, baseUrl };
        if (isGenerative) {
          provideEntry.resiliencyTests = { enable: "all" };
        }
        provides.push(provideEntry);
      }
    }

    // Consumes: any spec with a Mock port configured
    const portInput = screen.querySelector('.details .mock #mockPort');
    const portStr = portInput?.value?.trim();
    const portNum = portStr ? Number.parseInt(portStr, 10) : NaN;
    // Only add a consumes entry if mocks were run
    if (ranMock && !Number.isNaN(portNum) && portNum > 0) {
      consumes.push({ specification: specPath, port: portNum });
    }
  });

  return { provides, consumes };
}
window.addEventListener("load", () => Workspace.registerListeners());
if (CONFIG_ENABLED) {
  window.addEventListener("load", () => loadInitialConfigFromServerOnce());
}

async function loadInitialConfigFromServerOnce() {
  try {
    if (EphemeralStore.get('initial-config-loaded', false)) return;
    const { data, error } = await makeHttpCall("/config/load", { method: "GET" });
    if (error || !data) return;
    const provides = Array.isArray(data.provides) ? data.provides : [];
    const consumes = Array.isArray(data.consumes) ? data.consumes : [];
    EphemeralStore.set('initial-provides', provides);
    EphemeralStore.set('initial-consumes', consumes);
    EphemeralStore.set('initial-config-loaded', true);
    // Apply to any already-open screens
    try {
      const screens = document.querySelectorAll('.screen');
      screens.forEach(scr => applyInitialConfigToScreen(scr));
    } catch (_) { /* ignore */ }
  } catch (_) { /* ignore */ }
}

function applyInitialConfigToScreen(screen) {
  if (!screen || !screen.id) return;
  const specPath = screen.id;
  const fileType = screen.getAttribute('data-file-type');
  const provides = EphemeralStore.get('initial-provides', []);
  const consumes = EphemeralStore.get('initial-consumes', []);

  if (fileType === 'openapi') {
    const baseUrlInput = screen.querySelector('#testBaseUrl');
    if (typeof ensureBaseUrlInputState === 'function') ensureBaseUrlInputState(baseUrlInput);

    const match = provides.find(p => p && p.specification === specPath);
    if (match) {
      if (baseUrlInput) setBaseUrlInputValue(baseUrlInput, match.baseUrl ?? '', 'config');
      const genCheckbox = screen.querySelector('#generative');
      if (genCheckbox) genCheckbox.checked = !!(match.resiliencyTests && match.resiliencyTests.enable === 'all');
    }
  }

  // Apply mock port for any supported spec types that have #mockPort
  const consumeMatch = consumes.find(c => c && c.specification === specPath);
  if (consumeMatch) {
    const mockPortInput = screen.querySelector('.details .mock #mockPort');
    if (mockPortInput && !mockPortInput.value) mockPortInput.value = String(consumeMatch.port || '');
  }
}

// Expose for use in screen.js when creating screens
window.applyInitialConfigToScreen = applyInitialConfigToScreen;


document.addEventListener("DOMContentLoaded", function () {
  const sidebar = document.getElementById("left-sidebar");
  const triggerEls = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
  let tooltipList = triggerEls.map(el => new bootstrap.Tooltip(el));

  function updateTooltips() {
    const expanded = sidebar.getAttribute("aria-expanded") === "true";

    if (expanded) {
      // Dispose tooltips while sidebar expanded
      tooltipList.forEach(t => t.dispose());
      tooltipList = [];
    } else {
      // Reinitialize tooltips in collapsed mode
      tooltipList = triggerEls.map(el => new bootstrap.Tooltip(el));
    }
  }

  // Listen for sidebar toggle changes (aria-expanded attribute change)
  const observer = new MutationObserver(updateTooltips);
  observer.observe(sidebar, { attributes: true, attributeFilter: ["aria-expanded"] });

  // Run on page load too (in case sidebar defaults to expanded)
  updateTooltips();
});


async function resetWorkingDirectory(event) {
  if (event) event.preventDefault();
  const { data, error } = await makeHttpCall("/specifications/reset", {
    method: "POST"
  });
  if (error) {
    createAlert({
      title: "Reset operation failed.",
      message: error || response.statusText || "Unknown error",
      type: "error"
    });
    return;
  }
  createAlert({
    title: "Reset successful",
    message: "Your account has been reset successfully.",
    type: "success"
  });
}

document.getElementById('confirmResetBtn').addEventListener('click', async function() {
  // Hide the modal first
  const resetModal = bootstrap.Modal.getInstance(document.getElementById('resetConfirmationModal'));
  if (resetModal) resetModal.hide();

  await resetWorkingDirectory({ preventDefault: () => { } });
});

window.addEventListener('DOMContentLoaded', function() {
  const dropdownUserName = document.getElementById('dropdownUserName'); 
  if(!dropdownUserName) return;
  var name = dropdownUserName.textContent.trim();
  let initials = '';
  if (name) {
    const parts = name.split(' ');
    if (parts.length === 1 && parts[0].length >= 1) {
      initials = parts[0][0].toUpperCase();
    } else if(parts[0].length >= 1 && parts[parts.length-1].length >= 1) {
      initials = (parts[0][0] + parts[parts.length-1][0]).toUpperCase();
    }
  }
  document.getElementById('userInitialsIcon').textContent = initials || '';
});
