const proxyForm = document.querySelector("#proxyForm");
const startProxyBtn = document.querySelector("#startProxy");
const stopProxyBtn = document.querySelector("#stopProxy");
const proxyTable = document.querySelector("#proxyTable");
let proxyDrillDown;
let unsubscribeProxyTraffic;
let hasShownProxyStopSseAlert = false;

const PROXY_STOP_ALERT_DURATION = 5000;

const screen = proxyForm.closest(".screen");
const PROXY_INFO_SELECTOR = "#proxy > div"

const newTable = new TableMaker({
    screenId: "PROXY",
    target: proxyTable,
    groupBy: new Set(['path', 'method', 'response']),
    onDuplicate: ({ existingRow }) => {
        const countCell = existingRow.querySelector("td[data-key='count']");
        if (countCell) countCell.textContent = (parseInt(countCell.textContent || '0') + 1).toString();
        return false;
    },
    onAddingRow: ({ row, data }) => {
        proxyTable.classList.remove('hidden');
        const resp = data.response ?? data.status;
        row.setAttribute('data-result', classifyStatus(resp));
    }
});

const proxyCountsMap = (() => {
    const proxyScreen = document.querySelector('#proxy');
    if (!proxyScreen) return new Map();
    return getCountsMap(proxyScreen);
})();

function resetProxyCounts() {
    if (!proxyCountsMap || proxyCountsMap.size === 0) return;
    proxyCountsMap.forEach((value) => modifyChildSpanBy({ element: value, reset: true }));
}

function showProxyStoppedAlert({ message, type = "info" } = {}) {
    const alertMessage = message || "Stopped receiving proxy traffic events.";
    createAlert({
        title: "Proxy Stopped",
        message: alertMessage,
        type,
        duration: PROXY_STOP_ALERT_DURATION
    });
}

function classifyStatus(status) {
    const code = parseInt(status, 10);
    if (!Number.isFinite(code) || code <= 0) return 'error'; // connection or unknown error
    if (code >= 200 && code < 400) return 'success';
    if (code >= 400 && code < 600) return 'failed'; // 4xx and 5xx are failures
    return 'failed';
}

proxyForm?.addEventListener("submit", async (e) => {
    e.preventDefault();
    const formData = new FormData(proxyForm);
    const jsonData = Object.fromEntries(formData.entries());

    const { data, error } = await makeHttpCall("/proxy/start", {
        method: "POST",
        headers: {
            'Content-Type': 'application/json'
        },
        body: jsonData
    });

    if (error) {
        return createAlert({
            title: "Proxy Error",
            message: error,
            type: "error"
        });
    }

    createAlert({
        title: "Proxy Started",
        message: `${data.message}`,
        type: "success",
        duration: 5000
    });
    Workspace.updateInfoBox(screen, data.message, PROXY_INFO_SELECTOR);
    newTable.reset();
    proxyTable.classList.remove('hidden');
    resetProxyCounts();
    resetProxyFilter();
    showProxyCounts(true);
    disableAllProxyFilterButtons();

    subscribeProxyTraffic();
    disableProxyForm(data);
});

async function stopProxy() {
    const { data, error } = await makeHttpCall("/proxy/stop", {
        method: "POST",
        headers: {
            'Content-Type': 'application/json'
        },
        body: {}
    });

    if (error) {
        Workspace.updateInfoBox(screen, error, PROXY_INFO_SELECTOR);
        return createAlert({
            title: "Proxy Error",
            message: error,
            type: "error"
        });
    }

    Workspace.updateInfoBox(screen, null, PROXY_INFO_SELECTOR);
    showProxyStoppedAlert({ message: "Successfully stopped the proxy", type: "success" });

    createAlert({
        title: "Proxy Recording Success",
        message: "Captured recording will be available in proxy_generated.yaml.\nYou will find that file in the left sidebar.",
        type: "success",
        duration: 10000
    });

    enableProxyForm();
    if (data.generated) {
        addNode({ title: data.specName, path: data.specPath, type: "openapi", highlight: true });
        addNode({ title: data.examplesName, path: data.examplesPath, type: "folder", highlight: false });

        await waitForPath({ path: data.specPath, retryMs: 150, maxWait: 3000 })
            .then(node => { if (node) highlightAndSwitchTo({ node }); });
    }
}
window.addEventListener('unload', () => {
    makeHttpCall("/proxy/stop", {
        method: "POST",
        headers: {
            'Content-Type': 'application/json'
        },
        body: {}
    });
});

stopProxyBtn?.addEventListener("click", async (e) => {
    e.preventDefault();
    await stopProxy();
});

function setupProxyDrillDown() {
    try {
        const mainDiv = proxyTable?.parentElement;
        if (!mainDiv) return;
        proxyDrillDown = new DrillDown({
            mainDiv,
            table: proxyTable,
            eventId: 'proxy',
            groupColumns: new Set(['path', 'method', 'response']),
            onExtract: ({ key, cell }) => {
                const v = {}; v[key] = cell.getAttribute('data-value');
                const tr = cell.closest('tr');
                if (tr) {
                    const raw = tr.getAttribute('data-_originalpath');
                    if (raw) v._originalPath = raw;
                }
                return v;
            },
            onDrillDown: async ({ data }) => await getProxyDetails(data)
        });
        mainDiv.setAttribute('data-mode', 'table');
    } catch (e) { /* ignore */ }
}

async function updateProxyFormStatus() {
    const { data, error } = await makeHttpCall("/proxy/status", {
        method: "GET",
        headers: {
            'Content-Type': 'application/json'
        }
    });

    if (data.running === "true") {
        disableProxyForm(data);
        subscribeProxyTraffic();
        resetProxyCounts();
        resetProxyFilter();
        proxyTable.classList.remove('hidden');
        showProxyCounts(true);
        disableAllProxyFilterButtons();
    } else {
        enableProxyForm();
        unsubscribeProxyTraffic && unsubscribeProxyTraffic();
        // If there are rows from a previous session, ensure drilldown works when returning to the tab
        const hasRows = proxyTable?.querySelector('tbody > tr') !== null;
        if (hasRows && !proxyDrillDown) setupProxyDrillDown();
    }
}
function subscribeProxyTraffic() {
    if (unsubscribeProxyTraffic) unsubscribeProxyTraffic();
    hasShownProxyStopSseAlert = false;
    unsubscribeProxyTraffic = SseEventStreamer.subscribe({
        eventId: 'proxy-traffic',
        callbacks: {
            onData: (data) => {
                let payload;
                try { payload = (typeof data === 'string') ? JSON.parse(data) : data; } catch { return; }
                if (!proxyDrillDown) setupProxyDrillDown();
                const rawPath = payload.path;
                payload.path = normalizePath(payload.path);
                newTable.addRow({ path: payload.path, method: payload.method, response: payload.status, count: 1, _originalpath: rawPath });
                try { updateCounts({ countsMap: proxyCountsMap, result: classifyStatus(payload.status), by: 1 }); } catch(e) { /* noop */ }
                // enable filters just-in-time
                enableProxyFilterButton('total');
                const type = classifyStatus(payload.status);
                if (type === 'success') enableProxyFilterButton('success');
                if (type === 'failed') enableProxyFilterButton('failed');
                if (type === 'error') enableProxyFilterButton('error');
            },
            onError: () => {
                if (hasShownProxyStopSseAlert) return;
                hasShownProxyStopSseAlert = true;
                showProxyStoppedAlert();
            }
        }
    });
    window.addEventListener('beforeunload', () => unsubscribeProxyTraffic && unsubscribeProxyTraffic());
}

function enableProxyForm() {
    Workspace.updateInfoBox(screen, null, PROXY_INFO_SELECTOR);
    proxyForm.targetBaseURL.disabled = false;
    proxyForm.port.disabled = false;
    startProxyBtn.hidden = "";
    stopProxyBtn.hidden = "hidden";
}

function disableProxyForm(formData = null) {
    if (formData) {
        proxyForm.targetBaseURL.value = formData.targetBaseURL;
        proxyForm.port.value = formData.port;
    }
    proxyForm.targetBaseURL.disabled = true;
    proxyForm.port.disabled = true;
    startProxyBtn.hidden = "hidden";
    stopProxyBtn.hidden = "";
}

function normalizePath(path) {
    const segments = path.split('/').filter(Boolean);
    let paramCounter = 0;

    return '/' + segments.map(seg => {

        if (!isNaN(seg) && Number.isInteger(Number(seg))) {
            const name = paramCounter === 0 ? 'id' : `param${paramCounter}`;
            paramCounter++;
            return `{${name}}`;
        }

        return seg;
    }).join('/');
}

async function getProxyDetails(data) {
    const body = { path: (data._originalPath || data.path), method: data.method, status: parseInt(data.response, 10) };
    const { data: details, error } = await makeHttpCall('/proxy/data', { method: 'POST', body });
    if (error) {
        createAlert({ title: 'Failed to get proxy traffic details', message: error, type: 'error' });
        return null;
    }
    return details;
}

// Simple filter support (style identical, functionally filters table)
const proxyCountsList = document.querySelector('#proxy .counts');
if (proxyCountsList) {
    proxyCountsList.addEventListener('click', (e) => {
        const li = e.target?.closest('li');
        if (!li || !proxyCountsList.contains(li)) return;
        if (li.classList.contains('disabled')) return;
        const type = li.getAttribute('data-type');
        if (!type) return;
        const isReset = (proxyCountsList.getAttribute('data-filter') === type) || type === 'total';
        const newType = isReset ? 'total' : type;
        for (const el of proxyCountsList.querySelectorAll('[data-active]')) el.removeAttribute('data-active');
        proxyCountsList.setAttribute('data-filter', newType);
        proxyCountsList.querySelector(`li[data-type="${newType}"]`)?.setAttribute('data-active', 'true');
        applyProxyFilter(newType);
    });
}

function resetProxyFilter() {
    if (!proxyCountsList) return;
    for (const el of proxyCountsList.querySelectorAll('[data-active]')) el.removeAttribute('data-active');
    proxyCountsList.setAttribute('data-filter', 'total');
    proxyCountsList.querySelector('li[data-type="total"]')?.setAttribute('data-active', 'true');
    applyProxyFilter('total');
}

function applyProxyFilter(type) {
    const rows = Array.from(proxyTable?.querySelectorAll('tbody > tr') || []);
    let visibleCount = 0;
    for (const row of rows) {
        const res = row.getAttribute('data-result');
        const show = (type === 'total') || (res === type);
        row.style.display = show ? '' : 'none';
        if (show) visibleCount++;
    }
    if (proxyTable) proxyTable.classList.toggle('hidden', rows.length > 0 && visibleCount === 0);
    reflowProxyGroupedCells();
}

function reflowProxyGroupedCells() {
    if (!proxyTable || !newTable?.groupBy) return;

    const groupKeys = Array.from(newTable.groupBy);
    if (groupKeys.length === 0) return;

    const rows = Array.from(proxyTable.querySelectorAll('tbody > tr'));
    if (rows.length === 0) return;

    const groupMap = new Map();

    for (const row of rows) {
        let keySoFar = '';
        for (const key of groupKeys) {
            const cell = row.querySelector(`td[data-key="${key}"]`);
            if (!cell) continue;
            const value = cell.getAttribute('data-value') ?? '';
            keySoFar += `\u001F${value}`;
            if (!groupMap.has(keySoFar)) groupMap.set(keySoFar, []);
            groupMap.get(keySoFar).push({ row, cell });
        }
    }

    for (const entries of groupMap.values()) {
        const visibleEntries = entries.filter(({ row }) => row.style.display !== 'none');
        const hiddenEntries = entries.filter(({ row }) => row.style.display === 'none');

        if (visibleEntries.length === 0) {
            for (const { cell } of hiddenEntries) cell.removeAttribute('rowspan');
            continue;
        }

        const [first, ...rest] = visibleEntries;
        first.cell.classList.remove('hidden');
        if (visibleEntries.length > 1) first.cell.setAttribute('rowspan', String(visibleEntries.length));
        else first.cell.removeAttribute('rowspan');

        for (const { cell } of rest) {
            cell.classList.add('hidden');
            cell.removeAttribute('rowspan');
        }

        for (const { cell } of hiddenEntries) {
            cell.removeAttribute('rowspan');
        }
    }
}

// Hide counts initially; show on start/status
if (proxyCountsList) proxyCountsList.style.display = 'none';

function showProxyCounts(show) {
    if (!proxyCountsList) return;
    proxyCountsList.style.display = show ? '' : 'none';
}

function disableAllProxyFilterButtons() {
    if (!proxyCountsList) return;
    proxyCountsList.querySelectorAll('li.count').forEach(li => li.classList.add('disabled'));
}

function enableProxyFilterButton(type) {
    if (!proxyCountsList) return;
    proxyCountsList.querySelector(`li.count[data-type="${type}"]`)?.classList.remove('disabled');
}

window.addEventListener("load", async () => { await updateProxyFormStatus()  });
