// Spec Editor functionality using CodeMirror
// We'll load the CodeMirror components dynamically to avoid ES module issues
let EditorView, EditorState, basicSetup, oneDark, yaml;

// Function to dynamically import CodeMirror
async function loadCodeMirror() {
    const module = await import('./codemirror-bundle-yaml.esm.js');
    EditorView = module.EditorView;
    EditorState = module.EditorState;
    basicSetup = module.basicSetup;
    oneDark = module.oneDark;
    yaml = module.yaml;
}

// Map to store editor instances for each screen
const containerToEditor = new Map();

async function initSpecEditor(screen) {
    const specPath = screen.getAttribute('id');
    if (!specPath) {
        createAlert({ title: "Error", message: "No specification file selected", type: "error" });
        return;
    }

    const loader = screen.querySelector('.spec #spec-tab-loader');
    const specContainer = screen.querySelector('.spec > .container');
    await initSpecEditorInternal(specPath, loader, specContainer).then(() => {
        loadMermaidDiagram(screen, specContainer);
    });
}

function showLoader(loader, container) {
    if (loader) loader.style.display = 'block';
    if (container) container.style.display = 'none';
}

function hideLoader(loader, container) {
    if (loader) loader.style.display = 'none';
    if (container) container.style.display = 'block';
}

// Initialize the spec editor for a screen
async function initSpecEditorInternal(contentPath, loader, specContainer) {
    if (containerToEditor.has(specContainer)) return;
    // Load CodeMirror if not already loaded
    showLoader(loader, specContainer);
    if (!EditorView) await loadCodeMirror();

    try {
        const { data, error } = await makeHttpCall("/content/load", { method: "GET", queryParams: { path: contentPath } });
        if (error) {
            createAlert({ title: "Failed to load content", message: error, type: "error" });
            return;
        }

        hideErrors(specContainer);
        // Create editor container if it doesn't exist
        const editorContainer = specContainer.querySelector('.spec-editor');
        // Create the editor
        const editor = new EditorView({
            state: EditorState.create({
                doc: data,
                extensions: [
                    basicSetup,
                    yaml(),
                    oneDark
                ]
            }),
            parent: editorContainer
        });

        // Size editor now and also after it becomes visible
        resizeEditor(specContainer);
        scheduleResizeWhenVisible(specContainer);
        // Store the editor instance
        containerToEditor.set(specContainer, { editor, path: contentPath });

        // Add event listener to save button
        const saveButton = specContainer.querySelector('.savebtn.save');
        const bccButton = specContainer.querySelector('#bcc-test-btn');
        saveButton?.addEventListener('click', () => saveContent(specContainer, saveButton.getAttribute("data-validate")));
        bccButton?.addEventListener('click', () => testBackwardCompatibility(specContainer));
    } finally {
        hideLoader(loader, specContainer);
    }
}

// Save the spec file content
async function saveContent(container, validatePath) {
    const { editor, path } = containerToEditor.get(container) || {};
    if (!editor) return;

    hideErrors(container);
    const content = editor.state.doc.toString();
    if (validatePath) {
        const { error } = await makeHttpCall(`/validate/${validatePath}`, { method: "POST", body: { path, content } });
        if (error) {
            createAlert({ title: "Failed to save", message: "Content is invalid", type: "error" });
            showErrors(container, error);
            return
        }
    }

    const { error } = await makeHttpCall("/content/save", { method: "POST", body: { path, content } });
    if (error) {
        createAlert({ title: "Failed to save content", message: error, type: "error" });
        return;
    }

    createAlert({ title: "Contents Saved", message: "Contents saved successfully", type: "success", duration: 3000 });
    await loadMermaidDiagram(container, container);
}

async function testBackwardCompatibility(container) {
    const { editor, path } = containerToEditor.get(container) || {};
    if (!editor) return;

    hideErrors(container);
    const content = editor.state.doc.toString();
    const { error } = await makeHttpCall("/specifications/test-backward-compatibility", { method: "POST", body: { path, content } });

    if (error) {
        createAlert({ title: "Backward Compatibility Test", message: "Backward compatibility test failed", type: "error" });
        return showErrors(container, error);
    }

    createAlert({ title: "Backward Compatibility Test", message: "Changes are backward compatible", type: "success", duration: 3000 });
    return;
}

// Export functions to be used in other files
window.initSpecEditor = initSpecEditor;
// Initialize when the document is loaded
document.addEventListener('DOMContentLoaded', () => console.log('Spec editor module loaded'));

function hideErrors(container) {
    const bccErrorsRow = container.querySelector('.bcc-errors-row');
    const bccCardBody = bccErrorsRow?.querySelector('.card-body');
    const bccErrorsBtn = container.querySelector('.bcc-errors-btn');
    
    if (bccErrorsRow) {
        bccErrorsRow.style.display = 'none';
    }
    if (bccCardBody) {
        bccCardBody.innerHTML = '';
    }
    if (bccErrorsBtn) {
        bccErrorsBtn.setAttribute('aria-expanded', 'false');
        bccErrorsBtn.blur(); // Remove focus so border radius resets
    }
}

function showErrors(container, data) {
    hideErrors(container);
    
    const bccErrorsBtn = container.querySelector('.bcc-errors-btn');
    const bccErrorsRow = container.querySelector('.bcc-errors-row');
    const bccCardBody = bccErrorsRow?.querySelector('.card-body');
    
    if (!bccErrorsBtn || !bccErrorsRow || !bccCardBody) return;

    const addError = (text) => {
        const errorDiv = document.createElement('div');
        errorDiv.className = 'error-item';
        
        const preDiv = document.createElement('pre');
        preDiv.textContent = text;
        errorDiv.appendChild(preDiv);
        
        bccCardBody.appendChild(errorDiv);
    };

    let errorCount = 0;
    
    if (typeof data === 'string') {
        addError(data);
        errorCount = 1;
    } else if (typeof data === 'object' && data !== null) {
        Object.values(data).forEach(addError);
        errorCount = Object.keys(data).length;
    }
    
    // Update button with chevron on the left using your SVG file
    const errorText = errorCount === 1 ? 'Found 1 error' : `Found ${errorCount} errors`;
    bccErrorsBtn.innerHTML = `
        <img src="static/dropdown-chevron.svg" alt="Expand" class="chevron">
        <span>${errorText}</span>
    `;
    
    // Show the errors row
    bccErrorsRow.style.display = 'block';
    
    // Remove any existing event listeners to avoid conflicts
    bccErrorsBtn.onclick = null;
    
    // Use Bootstrap's collapse events instead of custom click handler
    const bccErrorsContent = container.querySelector('.bcc-errors-content');
    
    // Listen for Bootstrap collapse events
    bccErrorsContent.addEventListener('show.bs.collapse', function() {
        bccErrorsBtn.setAttribute('aria-expanded', 'true');
    });
    
    bccErrorsContent.addEventListener('hide.bs.collapse', function() {
        bccErrorsBtn.setAttribute('aria-expanded', 'false');
        bccErrorsBtn.blur(); // Remove focus when collapsing to reset border radius
    });
    
    // Set initial state
    bccErrorsBtn.setAttribute('aria-expanded', 'false');
}

function resizeEditor(container) {
    const editorHost = container.querySelector('.spec-editor');
    if (!editorHost) return;

    // Calculate height relative to the surrounding workspace, not the full window,
    // so the editor fills the .details area exactly like the Spec tab.
    const details = container.closest('.details');
    if (!details) return;

    // Toolbar row (Save button etc.) is the first direct .row under the container
    const rows = container.querySelectorAll(':scope > .row');
    const toolbarRow = rows.length > 0 ? rows[0] : null;
    const toolbarHeight = toolbarRow ? toolbarRow.getBoundingClientRect().height : 0;

    // Allow for small spacing/margins
    const extraSpacing = 16; // px
    const available = Math.max(200, details.clientHeight - toolbarHeight - extraSpacing);
    editorHost.style.height = `${available}px`;
}

// Resize all active editors on window resize for full-height experience
function resizeAllEditors() {
    for (const [container] of containerToEditor.entries()) {
        resizeEditor(container);
    }
}

window.addEventListener('resize', () => {
    // Debounce slightly by reusing requestAnimationFrame
    requestAnimationFrame(resizeAllEditors);
});

function scheduleResizeWhenVisible(container) {
    let attempts = 0;
    const tryResize = () => {
        attempts++;
        const details = container.closest('.details');
        if (details && details.clientHeight > 0) {
            resizeEditor(container);
            return;
        }
        if (attempts < 10) requestAnimationFrame(tryResize);
    };
    requestAnimationFrame(tryResize);
}
