// biome-ignore lint/complexity/noStaticOnlyClass:
class LocalStorage {
    /**
     * @param {string} key
     * @param {Object} defaultValue
     * @returns {Object}
     */
    static get(key, defaultValue = undefined) {
        const data = localStorage.getItem(key);
        return data ? JSON.parse(data) : defaultValue;
    }

    /**
     * @param {string} key
     * @param {Object} value
     */
    static set(key, value) {
        localStorage.setItem(key, JSON.stringify(value));
    }

    /**
     * @param {string} key
     * @returns {boolean}
     */
    static clear(key) {
        const containsKey = localStorage.getItem(key) !== null;
        localStorage.removeItem(key);
        return containsKey;
    }
}

// Ephemeral in-memory store for the current page lifecycle only.
// Does not persist across reloads or navigation.
class EphemeralStore {
    static #store = new Map();

    /**
     * @param {string} key
     * @param {any} defaultValue
     * @returns {any}
     */
    static get(key, defaultValue = undefined) {
        return this.#store.has(key) ? this.#store.get(key) : defaultValue;
    }

    /**
     * @param {string} key
     * @param {any} value
     */
    static set(key, value) {
        this.#store.set(key, value);
    }

    /**
     * @param {string} key
     * @returns {boolean}
     */
    static clear(key) {
        const had = this.#store.has(key);
        this.#store.delete(key);
        return had;
    }
}
