/**
 * @param {Object} options
 * @param {HTMLElement} options.screen
 * @param {HTMLElement} options.tab
 * @param {Boolean} options.withSelection
 * @param {Map<HTMLElement, TableMaker>} options.tableStore
 * @param {Map<HTMLElement, TableFilter>} options.filterStore
 * @param {Set<String>} options.groupBy=new Set(["path", "method", "response"])
 */
async function init({ screen, tab, withSelection, tableStore, filterStore, groupBy = new Set(["path", "method", "response"]) }) {
    if (tableStore.has(screen)) return;
    const countsMap = getCountsMap(tab);
    const table = tab.querySelector("table");
    if (!table) throw new Error("Table not found");

    const tableMaker = getOrPut({
        cache: tableStore,
        key: screen,
        defaultValue: () => new TableMaker({
            screenId: screen.id,
            target: table,
            groupBy: groupBy,
            onDuplicate: ({ existingRow, data }) => duplicateHandler({ existingRow, data, countsMap }),
            onAddingRow: ({ row, data }) => onNewRow({ row, tab, data, countsMap }),
            onExtract: extractCellValue,
            creator: rowCreator,
            addSelection: withSelection,
            onReset: ({ row }) => rowReset({ row })
        })
    });

    const tableFilter = getOrPut({
        cache: filterStore,
        key: screen,
        defaultValue: () => new TableFilter({
            countsList: tab.querySelector(".counts"),
            countsMap: countsMap,
            tableMaker: tableMaker,
            updateGroupCoverage: ({ group, coverage }) => handleCoverageHitEvent({ tableMaker, group, coverage })
        })
    });

    await resetTableWithApi({ screen, countsMap, tableMaker, tableFilter, target: tab.className });
}

/**
 * @param {Object} options
 * @param {HTMLElement} options.screen
 * @param {Map<string, HTMLElement>} options.countsMap
 * @param {TableMaker} options.tableMaker
 * @param {TableFilter} options.tableFilter
 * @param {String} options.target
 */
async function resetTableWithApi({ screen, countsMap, tableMaker, tableFilter, target }) {
    const { data, error } = await makeHttpCall("/specifications/endpoints", { method: "GET", queryParams: { specification: screen.id, target } });
    if (error) return createAlert({ title: `Failed to get endpoints for ${screen.id}`, message: error, type: "error" });

    tableMaker.reset();
    tableFilter.reset(tableMaker);
    tableMaker.target.classList.remove('hidden');
    setStyles({ table: tableMaker.target, id: target, countsMap, groupCount: tableMaker.groupBy.size });

    for (const endpoint of data.endpoints) {
        const data = { coverage: 0, ...endpoint, remark: undefined, result: undefined, _result: "notcovered", __skipRowResult: true }
        tableFilter.recordAndAddRow(data);
    }

    resetPreRequisiteError(screen);
    ensureTestBaseUrl(screen, data);
    resetCounts({ countsMap });
}

async function resetTableWithSelectionOrApi({ screen, countsMap, tableMaker, tableFilter, target }) {
    const selectedRows = tableMaker.getSelectedRows();
    let rowsToReset;
    if (!selectedRows || selectedRows.length === 0) {
        return await resetTableWithApi({ screen, countsMap, tableMaker, tableFilter, target });
    } else {
        rowsToReset = selectedRows;
    }

    for (const row of rowsToReset) resetCountsPerSelection({ countsMap, selection: row });
    tableMaker.reset(rowsToReset);
    tableFilter.reset(tableMaker, rowsToReset);
    tableMaker.target.classList.remove('hidden');
    setStyles({ table: tableMaker.target, id: target, countsMap, groupCount: tableMaker.groupBy.size });
    resetPreRequisiteError(screen);
}

function resetPreRequisiteError(screen) {
    const testTab = screen.querySelector('.details .test');
    if (!testTab) return;
    const errorAlert = testTab.querySelector('#prerequisite-error-alert');
    if (errorAlert) errorAlert.style.display = 'none';
}

function ensureTestBaseUrl(screen, data) {
    const testBaseURLInput = screen.querySelector("#testBaseUrl");
    if (typeof ensureBaseUrlInputState === 'function') ensureBaseUrlInputState(testBaseURLInput);
    if (!testBaseURLInput || !data.baseURL) return;
    if (typeof setBaseUrlInputValue === 'function') {
        setBaseUrlInputValue(testBaseURLInput, data.baseURL, 'spec');
    } else if (testBaseURLInput.value === "") {
        testBaseURLInput.value = data.baseURL;
    }
}

function resetCounts({ countsMap }) {
    countsMap.forEach((value, _) => modifyChildSpanBy({ element: value, reset: true }));
}

function resetCountsPerSelection({ countsMap, selection }) {
    const resultTd = selection.querySelector("td[data-key=result]");
    if (!resultTd) return;
    const counts = Array.from(resultTd.querySelectorAll(".count")).filter(count => count?.hasAttribute("data-value") === true);
    for (const count of counts) {
        const element = countsMap.get(count.getAttribute("data-key"));
        const lowerCountBy = Number.parseInt(count.getAttribute("data-value") || "0") || 0;
        modifyChildSpanBy({ element: element, count: -lowerCountBy });
    }
}

/**
 * @param {Object} options
 * @param {HTMLElement} options.screen
 * @param {HTMLElement} options.tab
 * @param {Map<HTMLElement, TableMaker>} options.tableStore
 * @param {Map<HTMLElement, DrillDown>} options.drillDownStore
 * @param {Map<HTMLElement, TableFilter>} options.filterStore
 * @param {Boolean} [options.needStart=true]
 * @param {function({exclusions: Object[], selections: Object[]}): Promise<{eventId: String}>} options.getEventId
 * @param {function({eventId: String, data: Object, filter: String?}): Promise<Object[]?>} options.onDrillDown
 * @param {Set<String>} options.groupBy=new Set(["path", "method", "response"])
 * @param {function(String): void} [options.onEnd]
 */
async function run({ screen, tab, tableStore, drillDownStore, filterStore, needStart = true, getEventId, onDrillDown, onEnd = () => {}, groupBy = new Set(["path", "method", "response"]) }) {
    const countsMap = getCountsMap(tab);
    const tableMaker = tableStore.get(screen);
    const tableFilter = filterStore.get(screen);

    if (!tableMaker) throw new Error("TableMaker not found");
    if (!tableFilter) throw new Error("TableFilter not found");

    const excluded = tableMaker.getExcluded();
    const currentSelections = tableMaker.getSelections();
    const tableSnapshot = tableMaker.snapshot();
    await resetTableWithSelectionOrApi({ screen, countsMap, tableMaker, tableFilter, target: tab.className })
    tableMaker.disableSelectionFeature();

    const mainDiv = tableMaker.target.parentElement;
    if (!mainDiv) throw new Error("MainDiv not found");

    let eventId;
    try {
        ({ eventId } = await getEventId({ exclusions: excluded, selections: currentSelections, snapshot: tableSnapshot }));
    } catch (e) {
        createAlert({ title: `Failed to start ${tab.className}`, message: e.message, type: "error" });
        toggleStart(tab, true);
        return;
    }

    const drillDown = getOrPut({
        cache: drillDownStore,
        key: screen,
        defaultValue: () => new DrillDown({
            mainDiv,
            table: tableMaker.target,
            eventId,
            groupColumns: groupBy,
            onDrillDown: ({ eventId, data }) => onDrillDown({ data, eventId, filter: tableFilter.getCurrentFilter() }),
            onExtract: extractCellValue
        })
    });
    drillDown.updateEventId(eventId);

    const unsubscribe = SseEventStreamer.subscribe({
        eventId: eventId,
        callbacks: {
            onData: (data) => onDataHandler({ screen, data, tableMaker, tableFilter }),
            onEnd: (data) => {
                createAlert({ title: data, type: "info", duration: 3000 });
                onComplete({ tab, tableMaker, tableFilter, countsMap });
                onEnd(eventId);
                unsubscribe(); 
            },
            onError: (data) => {
                onErrorHandler(screen, data.message);
                onComplete({ tab, tableMaker, tableFilter, countsMap });
                onEnd(eventId);
                unsubscribe();
            }
        }
    });

    if (needStart) {
        const { error } = await makeHttpCall("/task/start", { method: "POST", body: { id: eventId } });
        if (error) {
            unsubscribe();
            return createAlert({ title: `Failed to start ${tab.className}`, message: error, type: "error" });
        }
    }

    toggleStart(tab, false);
}

function onComplete({ tab, tableMaker, tableFilter, countsMap }) {
    toggleStart(tab, true);
    tableMaker.enableSelectionFeature();
    markNotCovered(tableMaker.target, countsMap);
    tableFilter.onEnd(tab.className === "test");
}

function toggleStart(tab, enable) {
    tab.querySelector("button.run").setAttribute("data-running", !enable);
}

/**
 * @param {Object} options
 * @param {Object} options.data
 * @param {TableMaker} options.tableMaker
 * @param {TableFilter} options.tableFilter
 */
function onDataHandler({ screen, data, tableMaker, tableFilter }) {
    const { type, ...payload } = data;
    switch (type) {
        case "ActuatorEndpointsEvent": {
            for (const endpoint of payload.missing) tableFilter.recordAndAddRow({ coverage: 0, ...endpoint, remark: "Missing In Spec", result: 0, _result: "notcovered" });
            break;
        }
        case "CoverageEvent": {
            tableFilter.addFinalCoverage(`${payload.coverage}%`)
            break;
        }
        case "CoverageHitEvent": {
            tableFilter.addCoverage(payload.group, payload.coverage);
            break;
        }
        case "ActuatorEvent": {
            tableFilter.setActuator(payload.enabled);
            break;
        }
        case "EndpointEvent": break;
        case "ArazzoMockDiagramEvent": break;
        default: tableFilter.recordAndAddRow(payload);
    }
}

function onErrorHandler(screen, message) {
    const testTab = screen.querySelector('.details .test');
    if (!testTab) return;
    const errorAlert = testTab.querySelector('#prerequisite-error-alert');
    const errorMessage = testTab.querySelector('#prerequisite-error-message');
    if (errorAlert && errorMessage) {
        errorMessage.innerHTML = message.replace(/\r\n\r\n/g, '\n').replace(/\n/g, '<br>');;
        errorAlert.style.display = 'flex';
    }
}

/**
 * @param {HTMLElement} tab
 * @returns {Map<string, HTMLElement>}
 */
function getCountsMap(tab) {
    const countsToElem = new Map();
    const counts = tab.querySelectorAll(".header > .counts > *");
    for (const count of counts) {
        const key = count.getAttribute("data-type");
        countsToElem.set(key, count);
    }
    return countsToElem;
}

/**
 * @param {Object} options
 * @param {String} options.key
 * @param {any} options.value
 * @param {Object} options.data
 * @returns {HTMLTableCellElement}
 */
function rowCreator({ key, value, data }) {
    const td = document.createElement("td");
    td.setAttribute('data-key', key);

    if (key === "workflow") {
        td.setAttribute('data-value', value);

        const flowChartBtn = document.createElement("button");
        flowChartBtn.classList.add("flow-chart-btn");
        flowChartBtn.classList.add("hidden");
        flowChartBtn.textContent = "Flow Chart";

        const textSpan = document.createElement("span");
        textSpan.textContent = value;

        td.appendChild(textSpan);
        td.appendChild(flowChartBtn);
        return td;
    }

    if (key !== "result") {
        td.setAttribute('data-value', value);
        td.textContent = value;
        return td;
    }

    const template = document.getElementById("count-fragment");
    if (!template) throw new Error("Template not found for result");
    const nodeFromTemplate = template.content.cloneNode(true);
    td.appendChild(nodeFromTemplate);

    if (!data._result || data.__skipRowResult) return td;
    increaseCellCount({ cell: td, result: data._result.toLowerCase(), by: data.result })
    return td;
}

function rowReset({ row }) {
    const remarkTd = row.querySelector("td[data-key=remark]");
    const resultTd = row.querySelector("td[data-key=result]");
    if (!remarkTd || !resultTd) throw new Error("Required table cell not found to reset");

    const template = document.getElementById("count-fragment");
    if (!template) throw new Error("Template not found for result");
    const nodeFromTemplate = template.content.cloneNode(true);

    remarkTd.setAttribute("data-value", "undefined");
    remarkTd.textContent = "";
    resultTd.replaceChildren(nodeFromTemplate);
}

/**
 * @param {Object} options
 * @param {HTMLTableRowElement} options.existingRow
 * @param {Object} options.data
 * @param {Map<string, HTMLElement>} options.countsMap
 * @returns {boolean}
 */
function duplicateHandler({ existingRow, data, countsMap }) {
    const remarkTdElem = existingRow.querySelector('td[data-key="remark"]');
    remarkTdElem?.setAttribute('data-value', data.remark);
    if (remarkTdElem && remarkTdElem.textContent.toLowerCase() != "covered") remarkTdElem.textContent = data.remark;

    const coverageTdElem = existingRow.querySelector('td[data-key="coverage"]');
    coverageTdElem?.setAttribute('data-value', data.coverage);
    if (coverageTdElem) coverageTdElem.textContent = `${data.coverage}%`;

    if (!data._result || data.__skipRowResult) return false;
    const resultTdElem = existingRow.querySelector('td[data-key="result"]');
    if (!resultTdElem) throw new Error("result td not found");
    increaseCellCount({ cell: resultTdElem, result: data._result, by: data.result });

    if (data.__skipCount) return false;
    updateCounts({ result: data._result, countsMap, by: data.result });
    return false;
}

/**
 * @param {Object} options
 * @param {HTMLTableRowElement} options.row
 * @param {HTMLElement} options.tab
 * @param {Map<string, HTMLElement>} options.countsMap
 * @param {Object} options.data
 * @returns {void}
 */
function onNewRow({ row, tab, countsMap, data }) {
    if (data._requestContentType) {
        const responseCell = row.querySelector('td[data-key="response"]');
        responseCell?.classList.add("response-cell")
        const span = document.createElement('span');
        span.textContent = data._requestContentType;
        responseCell?.appendChild(span);
    }

    if (data.__skipCount || !data.result) return;
    updateCounts({ result: data._result, countsMap, by: data.result });
}


function extractCellValue({ key, cell }) {
    const values = {}
    values[key] = cell.getAttribute('data-value');
    if (key === 'response' && cell.querySelector('span') !== null) values.requestContentType = cell.querySelector('span').textContent;
    return values
}

/**
 * @param {Object} options
 * @param {String} options.result
 * @param {Map<string, HTMLElement>} options.countsMap
 * @param {Number} options.by
 * @returns {void}
 */
function updateCounts({ countsMap, result, by }) {
    const countSpan = countsMap.get(result.toLowerCase());
    if (!countSpan) throw new Error(`Count span not found for ${result}`);
    modifyChildSpanBy({ element: countSpan, count: by });

    const totalSpan = countsMap.get('total');
    if (!totalSpan) throw new Error("Total span not found");
    modifyChildSpanBy({ element: totalSpan, count: by });
}

/**
 * @param {Object} options
 * @param {Element} options.element
 * @param {Number} [options.count=1]
 * @param {Boolean} [options.reset=false]
 */
function modifyChildSpanBy({ element, count = 1, reset = false }) {
    const countSpan = element.querySelector('span');
    if (!countSpan) throw new Error("Count span not found");
    const newValue = reset ? 0 : Math.max(Number.parseInt(countSpan.textContent || '0') + count, 0);
    if (element.classList.contains('count')) {
        if (newValue === 0) {
            element.classList.add('disabled');
        } else {
            element.classList.remove('disabled');
        }
    }
    countSpan.textContent = newValue.toString();
    countSpan.setAttribute('data-value', newValue.toString());
}

/**
 * @template T
 * @template U
 * @param {{cache: Map<T, U>, key: T, defaultValue: () => U}} options
 * @returns {U}
 */
function getOrPut({ cache, key, defaultValue }) {
    if (cache.has(key)) {
        const value = cache.get(key);
        if (value) return value;
    }

    const value = defaultValue();
    cache.set(key, value);
    return value;
}

/**
 * @param {Object} options
 * @param {HTMLTableElement} options.table
 * @param {String} options.id
 * @param {Map<string, HTMLElement>} options.countsMap
 */
function setStyles({ table, id, countsMap, groupCount }) {
    const cleanedId = id.replace(/^[^A-Za-z]+/, '')
    table.id = cleanedId;

    const style = document.createElement('style');
    style.textContent = `
#${cleanedId} > tbody > tr > td:nth-last-child(2) {
    text-transform: capitalize;
}

#${cleanedId} > tbody > tr > td[data-key="selection"]{
    opacity: 1;
    width: 0.5rem;
    &:not([disabled]) {
        pointer-events: all !important;
    }
}

#${cleanedId} > tbody > tr > td[data-key="coverage"] {
    width: 0.5rem;
}

#${cleanedId} > tbody > tr > td:nth-child(-n+${groupCount + 1}) {
    opacity: 1;
    background-color: rgb(var(--white));
    word-break: break-word;
    pointer-events: none;
}

#${cleanedId} > tbody > tr > td:nth-last-child(-n+${groupCount + 1}) {
    cursor: pointer;
}
    `;

    document.head.appendChild(style);
}

/**
 * @param {Object} options
 * @param {Element} options.cell
 * @param {String} options.result
 * @param {Number} options.by
 */
function increaseCellCount({ cell, result, by }) {
    const totalSpan = cell.querySelector('span[data-key="total"]');
    if (!totalSpan) throw new Error("Total span not found");
    const newTotal = Number.parseInt(totalSpan.getAttribute('data-value') || '0') + by;
    totalSpan.setAttribute('data-value', newTotal.toString());

    const countSpan = cell.querySelector(`span[data-key="${result.toLowerCase()}"]`);
    if (countSpan) {
        const newCount = Number.parseInt(countSpan.getAttribute('data-value') || '0') + by;
        countSpan.setAttribute('data-value', newCount.toString());
    }

    const nonTotalSpans = Array.from(cell.querySelectorAll('span.count:not([data-key="total"])'));
    const nonZeroCount = nonTotalSpans.filter(s => Number.parseInt(s.getAttribute('data-value') || '0') > 0).length;
    totalSpan.toggleAttribute('hidden', nonZeroCount <= 1);
}

/**
 * @param {Object} options
 * @param {TableMaker} options.tableMaker
 * @param {string} options.group
 * @param {number|string} options.coverage
 */
function handleCoverageHitEvent({ tableMaker, group, coverage }) {
    const rows = Array.from(tableMaker.tbody?.querySelectorAll("tr") || []);
    const mainGroupKey = tableMaker.getMainGroupKey();
    const [first, ...rest] = rows.filter((tr) => {
        return tr.querySelector(`td[data-key=${mainGroupKey}]`)?.getAttribute("data-value") === group;
    });

    if (first === undefined) return;
    for (const row of [first, ...rest]) {
        const coverageTd = row.querySelector('td[data-key="coverage"]');
        if (!coverageTd) throw new Error("Coverage td not found");
        coverageTd.classList.add('hidden');
        coverageTd.removeAttribute('data-main');
        coverageTd.textContent = typeof coverage === "number" ? `${coverage}%` : coverage;
    }

    const firstCoverageTd = first.querySelector('td[data-key="coverage"]');
    if (!firstCoverageTd) throw new Error("First coverage td not found");
    firstCoverageTd.classList.remove('hidden');
    firstCoverageTd.setAttribute('data-main', 'true');
    firstCoverageTd.setAttribute('rowspan', (rest.length + 1).toString());
}

function markNotCovered(table, countsMap) {
    const rows = Array.from(table.querySelectorAll('tbody > tr')).filter(tr => tr.querySelector('td[data-key="remark"]').getAttribute('data-value') === "undefined");
    let excludedCount = 0;
    let notCoveredCount = 0;

    for (const row of rows) {
        const remarkTd = row.querySelector('td[data-key="remark"]');
        const resultTd = row.querySelector('td[data-key="result"]');
        if (!remarkTd || !resultTd) throw new Error("Required table cells not found");

        const isExcluded = row.getAttribute("disabled") === "true";
        const label = isExcluded ? "Excluded" : "Not Covered";
        const key = isExcluded ? "excluded" : "notcovered";

        remarkTd.setAttribute('data-value', key);
        remarkTd.textContent = label;
        increaseCellCount({ cell: resultTd, result: key, by: 1 });
        isExcluded ? excludedCount++ : notCoveredCount++;
    }

    if (excludedCount > 0) updateCounts({ result: "excluded", countsMap, by: excludedCount });
    if (notCoveredCount > 0) updateCounts({ result: "notcovered", countsMap, by: notCoveredCount });
}
