class TableMaker {

    /**
     * @param {Object} options
     * @param {HTMLTableElement} options.target
     * @param {Set<string>} options.groupBy
     * @param {Boolean} [options.addSelection="false"]
     * @param {function({key: string, value: any, data: Object}): HTMLTableCellElement} [options.creator]
     * @param {function({existingRow: HTMLTableRowElement, data: Object}): Boolean} [options.onDuplicate]
     * @param {function({row: HTMLTableRowElement, data: Object}): void} [options.onAddingRow]
     * @param {function({key: String, cell: HTMLTableCellElement}): Object} [options.onExtract]
     * @param {function({row: HTMLTableRowElement}): void} [options.onReset]
     */
    constructor({ screenId, target, groupBy, addSelection = false, creator, onDuplicate, onAddingRow, onExtract, onReset }) {
        this.id = self.crypto.randomUUID()
        this.screenId = screenId;
        this.target = target
        this.groupBy = groupBy;
        this.onAddingRow = onAddingRow
        this.addSelection = addSelection;
        this.newRowSelection = addSelection;
        this.onReset = onReset|| (() => {});
        this.onDuplicate = onDuplicate || (() => true);
        this.onExtract = onExtract || (({ cell }) => cell.getAttribute('data-value'));
        this.creator = creator || (({ value }) => {
            const td = document.createElement("td");
            td.textContent = value;
            return td;
        });

        this._abortController = new AbortController();
        this.columns = Array.from(target?.querySelectorAll("th") || []).map(td => td.innerText.trim());
        this.mainGroups = new Map();
        this.filteredSelections = new Map(LocalStorage.get(`${screenId}-exclusions`, []));

        if (target.querySelector("tbody")) {
            this.tbody = target.querySelector("tbody");
        } else {
            this.tbody = document.createElement("tbody");
            target.appendChild(this.tbody);
        }

        if (!this.addSelection) return;
        this.filterHeader = this.target.parentElement.parentElement.parentElement.querySelector('.filterHeader');
        this.errorMessageDiv = this.filterHeader.querySelector('.error');
        this.filterButton = this.filterHeader.querySelector('.filter');
        this.clearButton = this.filterHeader.querySelector('.clear');
        this._setupCheckBoxes()
    }

    _setupCheckBoxes() {
        const signal = this._abortController.signal;
        this.filterButton.addEventListener('click', () => this.filterSelected(), { signal });
        this.clearButton.addEventListener('click', () => this.clearSelected(), { signal });
        this.tbody.addEventListener('change', (e) => {
            if (e.target.type === 'checkbox') this._updateFilterButtons()
        }, { signal });

        const hasSelectionTh = this.target.querySelector("thead > tr > th:first-child")?.getAttribute('data-key') === 'selection';
        if (hasSelectionTh) return;

        const thR = this.target.querySelector("thead > tr");
        const wrapper = document.createElement("th");
        const input = document.createElement("input");
        input.type = "checkbox";
        input.addEventListener("change", () => this._flipSelections(input), { signal });
        wrapper.appendChild(input);
        wrapper.setAttribute('data-key', 'selection');
        thR?.insertBefore(wrapper, thR.firstChild);
    }

    destroy() {
        this._abortController.abort();
        this.target.querySelector("thead > tr > th:first-child[data-key='selection']")?.remove();
        LocalStorage.clear(`${this.screenId}-exclusions`);
    }

    reset(rows) {
        if (!rows || rows.length == 0) {
            this.mainGroups.clear();
            this.target.querySelector("tbody")?.replaceChildren();
            return;
        }
        for (const row of rows) this.onReset({ row });
    }

    _getSelectedRows() {
        return Array.from(this.tbody.querySelectorAll("tr")).filter(row => row.querySelector("input[type=checkbox]")?.checked === true);
    }

    _getStatsPerGroup() {
        if (!this.groupBy?.size) return new Map();

        const statsMap = new Map();
        for (const key of this.groupBy) {
            statsMap.set(key, { total: 0, enabled: 0, disabled: 0, values: new Map() });
        }

        const rows = this.tbody?.querySelectorAll("tr") ?? [];
        for (const row of rows) {
            const isRowDisabled = row.hasAttribute('disabled');
            let entryKey = "";

            for (const key of this.groupBy) {
                const cell = row.querySelector(`td[data-key="${key}"]`);
                if (!cell) continue;

                const stats = statsMap.get(key);
                const value = cell.getAttribute('data-value');
                entryKey = `${entryKey}+${key}:${value}`;

                let entry = stats.values.get(entryKey);
                if (!entry) {
                    entry = { isDisabled: isRowDisabled, mainTd: cell.hasAttribute("data-main") ? cell : undefined };
                    stats.values.set(entryKey, entry);
                } else {
                    entry.isDisabled &&= isRowDisabled;
                    entry.mainTd ||= cell.hasAttribute("data-main") ? cell : undefined;
                }
            }
        }

        for (const stats of statsMap.values()) {
            for (const { isDisabled, mainTd } of stats.values.values()) mainTd?.toggleAttribute('disabled', isDisabled);
            stats.total = stats.values.size;
            stats.disabled = Array.from(stats.values.values()).filter(e => e.isDisabled).length;
            stats.enabled = stats.total - stats.disabled;
            delete stats.values;
        }

        return statsMap;
    }

    _updateTableHeaderCounts(isPlayBack = false) {
        const statsMap = this._getStatsPerGroup();
        const headers = this.target.querySelectorAll('thead > tr > th[data-key]');

        if (isPlayBack) return;
        for (const th of headers) {
            const dataKey = th.getAttribute('data-key');
            const stats = statsMap.get(dataKey);
            if (!stats) continue;

            th.setAttribute('data-total', stats.total);
            th.setAttribute('data-enabled', stats.enabled);
            th.setAttribute('data-disabled', stats.disabled);
        }
    }

    _updateFilterSelections(key, data) {
        this.filteredSelections.set(key, data);
        LocalStorage.set(`${this.screenId}-exclusions`, Array.from(this.filteredSelections));
    }

    _clearFilterSelection(key) {
        this.filteredSelections.delete(key);
        LocalStorage.set(`${this.screenId}-exclusions`, Array.from(this.filteredSelections));
    }

    _updateFilterButtons() {
        const selectedRows = this._getSelectedRows();
        if (selectedRows.length === 0) {
            this.filterHeader.setAttribute("data-active", "false");
            this.filterButton.setAttribute("data-active", "false");
            this.clearButton.setAttribute("data-active", "false");
            return;
        }

        this.filterHeader.setAttribute("data-active", "true");
        const canAnyBeExcluded = selectedRows.some(row => !row.hasAttribute('disabled'));
        const canAnyBeIncluded = selectedRows.some(row => row.hasAttribute('disabled'));

        if (canAnyBeExcluded && canAnyBeIncluded) {
            this.errorMessageDiv.setAttribute("data-active", "true");
            this.filterButton.setAttribute("data-active", "false");
            this.clearButton.setAttribute("data-active", "false");
            return;
        }

        this.errorMessageDiv.setAttribute("data-active", "false");
        this.filterButton.setAttribute("data-active", canAnyBeExcluded ? "true" : "false")
        this.clearButton.setAttribute("data-active", canAnyBeIncluded ? "true" : "false")
    }

    filterSelected() {
        const selectedRows = this._getSelectedRows();

        for (const selectedRow of selectedRows) {
            const data = this._extractRowValues(selectedRow);
            const key = this.getKeyFor(data);
            this._updateFilterSelections(key, data);
            selectedRow.setAttribute("disabled", "true");
        }

        this._updateAll(false);
        this._updateFilterButtons();
        this._updateTableHeaderCounts();
    }

    clearSelected() {
        const selectedRows = this._getSelectedRows();

        for (const selectedRow of selectedRows) {
            const data = this._extractRowValues(selectedRow);
            const key = this.getKeyFor(data);
            this._clearFilterSelection(key);
            selectedRow.removeAttribute("disabled")
        }

        this._updateAll(false);
        this._updateFilterButtons();
        this._updateTableHeaderCounts();
    }

    _updateAll(checked) {
        const checkboxes = Array.from(this.target.querySelectorAll("input[type=checkbox]"));
        for (const checkbox of checkboxes) checkbox.checked = checked;
    }

    _flipSelections(input) {
        const checkboxes = Array.from(this.target.querySelectorAll("tr:not([disabled]) input[type=checkbox]"));
        for (const checkbox of checkboxes) checkbox.checked = input.checked;
        this._updateFilterButtons();
    }

    /**
     * @param {HTMLElement} target
     */
    _addSelection(target) {
        if (!this.addSelection) return;
        const td = document.createElement("td");
        const input = document.createElement("input");
        input.type = "checkbox";
        if (!this.newRowSelection) td.setAttribute('disabled', "true");
        td.setAttribute('data-key', 'selection');
        td.appendChild(input);
        target.insertBefore(td, target.firstChild);
    }

    /**
     * @param {HTMLTableRowElement} row
     */
    _extractRowValues(row) {
        let values = {};
        for (const column of this.groupBy) {
            const cell = row.querySelector(`td[data-key="${column}"]`);
            values = { ...values, ...this.onExtract({ key: column, cell: cell }) };
        }
        return values;
    }

    getExcluded() {
        if (!this.addSelection) return [];
        return Array.from(this.filteredSelections.values());
    }

    getSelections() {
        const selectedRows = this._getSelectedRows();
        return selectedRows.map(row => this._extractRowValues(row));
    }

    getSelectedRows() {
        return this._getSelectedRows();
    }

    snapshot() {
      return Array.from(this.tbody.querySelectorAll("tr")).map(row => {
        const data = this._extractRowValues(row);
        const totalCount = Number.parseInt(row.querySelector("td[data-key=result] > .count[data-key=total]")?.getAttribute("data-value") || "0");
        const notCoveredCount = Number.parseInt(row.querySelector("td[data-key=result] > .count[data-key=notcovered]")?.getAttribute("data-value") || "0");
        const excludedCount = Number.parseInt(row.querySelector("td[data-key=result] > .count[data-key=excluded]")?.getAttribute("data-value") || "0");
        const isExcluded = row.getAttribute("disabled") === "true";
        const count = totalCount - excludedCount - notCoveredCount;
        return { ...data, count, isExcluded };
      });
    }

    enableSelectionFeature() {
        if (!this.addSelection) return;
        this.newRowSelection = true;
        const thR = this.target.querySelector("thead > tr > th:first-child")?.removeAttribute('disabled');
        for (const row of this.tbody?.querySelectorAll("tr") || []) {
            row.querySelector("td:first-child")?.removeAttribute('disabled');
        }
    }

    disableSelectionFeature() {
        if (!this.addSelection) return;
        this.newRowSelection = false;
        this.filterHeader.setAttribute("data-active", "false");
        this.target.querySelector("thead > tr > th:first-child")?.setAttribute("disabled", "true");
        for (const row of this.tbody?.querySelectorAll("tr") || []) {
            row.querySelector("td:first-child")?.setAttribute("disabled", "true");
        }
    }

    applyOnEach(fn) {
        for (const row of this.tbody?.querySelectorAll("tr") || []) {
            fn(row);
        }
    }

    /**
     * @param {Object<string, any>} data
    */
    addRow(data, isPlayBack = false) {
        const shouldProceed = this._handleIfDuplicate(data);
        if (!shouldProceed) return;

        const insertBeforeOptions = [];
        const tr = document.createElement("tr");
        this._addSelection(tr);
        let keySoFar = "";

        for (const [key, value] of Object.entries(data)) {
            if (key.startsWith("__")) continue;
            if (key.startsWith("_")) {
                tr.setAttribute(`data-${key}`, value);
                continue;
            }

            const td = this.creator({ key, value, data });
            td.setAttribute("data-key", key);
            td.setAttribute("data-value", value);

            if (this.groupBy.has(key)) {
                keySoFar += `-${value}`;
                this._updateElseAdd({ keySoFar, currentTableData: td, currentTableRow: tr, insertBeforeOptions });
            }

            tr.appendChild(td);
        }

        this.onAddingRow?.({ row: tr, data });
        const finalInsertBefore = insertBeforeOptions.filter(t => t).pop();
        this._insertRow(isPlayBack, tr, finalInsertBefore);
    }

    async _insertRow(isPlayBack, tr, insertBefore) {
        this._applyFilterIfApplicable(tr);
        if (!isPlayBack) {
            requestAnimationFrame(() => {
                this.tbody?.insertBefore(tr, insertBefore?.nextSibling)
                this._updateTableHeaderCounts(isPlayBack);
            });
        } else {
            this.tbody?.insertBefore(tr, insertBefore?.nextSibling);
            this._updateTableHeaderCounts(isPlayBack);
        }
    }

    _applyFilterIfApplicable(row) {
        if (!this.addSelection) return;
        if (this.filteredSelections.length === 0) return;

        const rowData = this._extractRowValues(row);
        const key = this.getKeyFor(rowData);
        if (!this.filteredSelections.has(key)) return;

        this._updateFilterSelections(key, rowData);
        row.setAttribute("disabled", "true");
    }

    _isExcluded(data) {
        if (!this.addSelection) return false;
        if (this.filteredSelections.size === 0) return false;
        const key = this.getKeyFor(data);
        return this.filteredSelections.has(key)
    }

    /**
     * @param {Object<string, any>} data
     * @returns {boolean}
    */
    _handleIfDuplicate(data) {
        const finalKey = this.getKeyFor(data);
        if (this.mainGroups.has(finalKey)) {
            return this.onDuplicate({ existingRow: this.mainGroups.get(finalKey).lastRow, data });
        }

        return true;
    }

    getKeyFor(data) {
        return `-${Object.entries(data).filter(([key]) => this.groupBy.has(key)).map(([_, value]) => value).join("-")}`
    }

    /**
     * @param {Object<string, any>} data
     * @returns {Object<string, string>}
     */
    getMainGroupEntry(data) {
        const mainKey = Array.from(this.groupBy)[0];
        const value = data[mainKey];
        return { key: mainKey, value };
    }

    /**
     * @returns {string}
     */
    getMainGroupKey() {
        return Array.from(this.groupBy)[0];
    }

    /**
     * @param {Object} params
     * @param {string} params.keySoFar
     * @param {HTMLElement} params.currentTableData
     * @param {HTMLElement} params.currentTableRow
     * @param {Array<HTMLElement>} params.insertBeforeOptions
     */
    _updateElseAdd({ keySoFar, currentTableData, currentTableRow, insertBeforeOptions }) {
        const existing = this.mainGroups.get(keySoFar);
        if (existing) {
            existing.td.setAttribute("rowspan", ++existing.count);
            currentTableData.classList.add("hidden");
            insertBeforeOptions.push(existing.lastRow);
            existing.lastRow = currentTableRow;
        } else {
            currentTableData.setAttribute("data-main", "true");
            this.mainGroups.set(keySoFar, { count: 1, td: currentTableData, lastRow: currentTableRow });
        }
    }

    _modifyGroup({ data, shared }) {
        let keySoFar = "";
        for (const [key, value] of Object.entries(data).filter(([key]) => this.groupBy.has(key))) {
            keySoFar += `-${value}`;
            const existing = this.mainGroups.get(keySoFar);
            if (!existing) throw new Error(`Existing not found for ${keySoFar}`);

            const newRowSpan = Number.parseInt(existing.td.getAttribute("rowspan") || "0") - 1;
            existing.td.setAttribute("rowspan", newRowSpan);
            existing.count = newRowSpan;
            existing.lastRow = existing.lastRow.previousElementSibling;

            if (shared.has(key)) {
                const selector = existing.td.parentElement.querySelector(`td[data-key="${shared.get(key)}"]`);
                selector.setAttribute("rowspan", newRowSpan);
            }

            if (newRowSpan < 0) this.mainGroups.delete(keySoFar);
        }
    }
}
