/** @type {Map<HTMLElement, TableMaker>} */
const screenToTestTableMaker = new Map();
/** @type {Map<HTMLElement, DrillDown>} */
const screenToTestDrillDown = new Map();
/** @type {Map<HTMLElement, TableFilter>} */
const screenToTableFilter = new Map();
/** @type {Map<HTMLElement, String>} */
const screenToTestId = new Map();

function clearTestFor(screen) {
    if (!screenToTestTableMaker.has(screen)) return;
    screenToTestTableMaker.get(screen).destroy();
    screenToTestTableMaker.delete(screen);
    screenToTestId.delete(screen);
    initTest(screen);
}

function initTest(screen) {
    init({
        screen,
        tab: screen.querySelector(".details .test"),
        withSelection: true,
        tableStore: screenToTestTableMaker,
        filterStore: screenToTableFilter
    });
}

async function runTest(screen) {
    const path = screen.id;
    const baseUrl = screen.querySelector("#testBaseUrl").value.trim();
    if (!URL.parse(baseUrl)) return createAlert({ title: "Invalid URL", message: "Please enter a valid URL", type: "error" });

    const testTab = screen.querySelector('.details .test');
    if (testTab) {
        const errorAlert = testTab.querySelector('#prerequisite-error-alert');
        if (errorAlert) {
            errorAlert.style.display = 'none';
        }
    }

    run({
        screen,
        tab: screen.querySelector(".details .test"),
        tableStore: screenToTestTableMaker,
        drillDownStore: screenToTestDrillDown,
        filterStore: screenToTableFilter,
        getEventId: async({ exclusions, selections, snapshot }) => await getTestEventId(screen, path, baseUrl, exclusions, selections, snapshot),
        onDrillDown: getTestDetails
    })
}

/**
 * @param {HTMLElement} screen
 * @param {String} path
 * @param {String} baseUrl
 * @param {Object[]} exclusions
 * @param {Object[]} selections
 * @returns {Promise<{eventId: String}>}
 */
async function getTestEventId(screen, path, baseUrl, exclusions, selections, snapshot) {
    const isGenerative = screen.querySelector('#generative')?.checked || false;
    const previousTestId = selections.length !== 0 ? screenToTestId.get(screen) : null;
    const currentSnapShot = selections.length !== 0 ? snapshot : null;
    const data = { path, testBaseUrl: baseUrl.trim(), generative: isGenerative, exclusions, selections, previousTestId, snapshot: currentSnapShot };

    const { data: eventData, error } = await makeHttpCall("/test", { method: "POST", body: data });
    if (error) {
        screen.querySelector(".details .test button.run").setAttribute("data-running", "false");
        createAlert({ title: "Failed to run tests", message: error, type: "error" });
        throw new Error(error);
    }

    try { EphemeralStore.set(`ran-test-${screen.id}`, true); } catch (_) { /* ignore */ }
    screenToTestId.set(screen, eventData.id);
    return { eventId: eventData.id };
}

/**
 * @param {Object} options
 * @param {String} options.eventId
 * @param {Object} options.data
 * @param {String?} options.filter
 * @returns {Promise<Object[]?>}
 */
async function getTestDetails({ eventId, data, filter }) {
    const { data: details, error } = await makeHttpCall(`/test/${eventId}`, { method: "POST", body: data });
    if (error) {
        createAlert({ title: "Failed to get endpoint details", message: error, type: "error" });
        return null;
    }

    if (!filter || filter === "total") return details;
    return details.filter(d => d.result.toLowerCase() === filter.toLowerCase());
}
