const ServerPickerTemplate = document.createElement('template');
ServerPickerTemplate.innerHTML = `
<div class="servers-btn">
    <span>Servers</span>
    <span class="servers icon-servers" aria-label="Servers"></span>
</div>
<dialog>
    <div class="dialog-content">
        <p>Pick the server BaseURL to use for each API Specification</p>
        <div class="servers-list"></div>
        <button type="button" class="close-dialog-btn" aria-label="Close">Close</button>
    </div>
</dialog>
`;

class ServerPicker extends HTMLElement {
    /**@type {string} */
    #specFile;
    /**@type {HTMLSpanElement} */
    #serversSvgBtn;
    /**@type {HTMLDialogElement} */
    #dialog;
    /**@type {Array<{name: string, path: string, relativePath: string, servers: Array<[string?, string?]>}>} */
    #servers = [];
    /**@type {{[key: string]: string}} */
    #apiNameToSelectedServer = {};

    constructor() {
        super();
        this.appendChild(ServerPickerTemplate.content.cloneNode(true));
        const specFileFromNearestScreen = this.closest(".screen")?.getAttribute("id");
        if (!specFileFromNearestScreen) throw new Error("ServerPicker must be inside an element with class 'screen' and have an id attribute representing the spec file.");
        this.#specFile = specFileFromNearestScreen;
        // @ts-ignore
        this.#serversSvgBtn = this.querySelector('.servers-btn');
        // @ts-ignore
        this.#dialog = this.querySelector('dialog');
    }

    getServers() {
        return this.#apiNameToSelectedServer;
    }

    async connectedCallback() {
        this.#serversSvgBtn.addEventListener('click', this.#openDialog.bind(this));
        this.#dialog.addEventListener('keydown', this.#handleKeydown.bind(this));
        this.#dialog.querySelector('.close-dialog-btn')?.addEventListener('click', this.#closeDialog.bind(this));
        this.#dialog.addEventListener('click', this.#handleOutsideClick.bind(this));
        this.#listenForScreenChange();
    }

    #listenForScreenChange() {
        const handler = async (event) => {
            if (event.detail?.screenId !== this.#specFile) return;
            await this.#renderServers();
            this.#listenForServerChanges();
            document.removeEventListener('screenChanged', handler);
        };
        document.addEventListener('screenChanged', handler);
    }

    #openDialog() {
        this.#dialog.showModal();
    }

    #closeDialog() {
        this.#dialog.close();
    }

    #handleKeydown(event) {
        if (event.key === 'Escape') {
            this.#closeDialog();
        }
    }

    #handleOutsideClick(event) {
        const insideDialogContent = event.target.closest(".dialog-content");
        if (!insideDialogContent) this.#closeDialog();
    }

    async #renderServers() {
        await this.#fetchServers();
        const docFrag = document.createDocumentFragment();
        for (const serverInfo of this.#servers) {
            const openApiServer = this.#renderServerSelector(serverInfo);
            docFrag.appendChild(openApiServer);
        }
        const serversList = this.#dialog.querySelector('.servers-list');
        serversList?.replaceChildren(docFrag);
    }

    #renderServerSelector(serverInfo) {
        const container = document.createElement("div");
        container.classList.add("server-container");

        const title = document.createElement("p");
        title.textContent = serverInfo.name;
        container.appendChild(title);

        const filePath = document.createElement("span");
        filePath.classList.add("file-path");
        filePath.textContent = serverInfo.relativePath;
        container.appendChild(filePath);

        const selector = document.createElement("select");
        selector.classList.add("server-selector");
        selector.setAttribute("aria-label", `Select server for ${serverInfo.name}`);

        const serverEntries = serverInfo.servers || [];
        serverEntries.forEach(([name, url], index) => {
            const option = document.createElement("option");
            option.value = url;
            option.textContent = name && name.trim() ? `${name} - ${url}` : url;
            if (index === 0) option.selected = true;
            selector.appendChild(option);
        });

        container.appendChild(selector);
        this.#apiNameToSelectedServer[serverInfo.name] = serverEntries[0]?.[1] || null;
        selector.addEventListener("change", () => this.#apiNameToSelectedServer[serverInfo.name] = selector.value);
        return container;
    }

    async #fetchServers() {
        const { data, error } = await makeHttpCall("/workflow/servers", { queryParams: { specPath: this.#specFile }, method: "GET" });
        if (error) return createAlert({ title: "Could not fetch servers for spec file", message: error, type: "error", duration: 5000 });
        this.#servers = data || [];
    }

    #listenForServerChanges() {
        SseEventStreamer.subscribe({
            eventId: "WATCHER",
            callbacks: {
                onData: async (data) => {
                    if (data.eventKind !== "ENTRY_MODIFY") return;
                    const serverSpec = this.#servers.find(s => s.path === data.node.path);
                    if (!serverSpec) return;
                    await this.#renderServers();
                }
            }
        });
    }
}

customElements.define('server-picker', ServerPicker);
