/**
 * @typedef {"success" | "failed" | "error" | "notcovered" | "excluded" | "total"} ResultType
 */

class TableFilter {
    /**
     * @param {Object} options
     * @param {HTMLOListElement} options.countsList
     * @param {Map<string, HTMLElement>} options.countsMap
     * @param {TableMaker} options.tableMaker
     * @param {(arg: { group: string, coverage: number }) => void} options.updateGroupCoverage
     */
    constructor({ countsList, countsMap, tableMaker, updateGroupCoverage }) {
        this.countsList = countsList;
        this.countsMap = countsMap;
        this.tableMaker = tableMaker;
        this.updateGroupCoverage = updateGroupCoverage;

        this._events = [];
        /** @type {Map<string, number>} */
        this._coverage = new Map();
        /** @type {ResultType|null} */
        this.savedFilter = null;
        this.countsList.addEventListener('click', this._onClick.bind(this));
    }

    /** @param {MouseEvent} event */
    _onClick(event) {
        const li = event.target?.closest('li');
        if (!li) return;

        const type = li.getAttribute("data-type");
        const count = Number(li.querySelector('span:last-child')?.getAttribute("data-value") || "0");
        if (!type || count === 0) return createAlert({ title: "Cannot Filter", message: "No data to filter on", type: "info", duration: 1500 });

        const isReset = type === "total" || this.countsList.getAttribute("data-filter") === type;
        this._applyFilter(isReset ? "total" : type);
    }

    /**
     * @param {ResultType} type
     */
    _applyFilter(type) {
        const li = this.countsList.querySelector(`[data-type="${type}"]`);
        for (const el of this.countsList.querySelectorAll('[data-active]')) el.removeAttribute('data-active');
        if (li) li.setAttribute('data-active', 'true');
        this.countsList.setAttribute("data-filter", type);

        const rows = this.getFilteredEvents(type);
        requestAnimationFrame(() => {
            this.tableMaker.reset();
            for (const row of rows) this.tableMaker.addRow(row, true);
            this._updatePaths(rows);
        });
    }

    /**
     * @param {ResultType} type
     * @returns {Object[]}
     */
    getFilteredEvents(type) {
        return type === "total" ? this._events : this._events.filter(e => e.__normalizedResult === type);
    }

    /**
     * @param {Object} data
     */
    recordAndAddRow(data) {
        const result = this._normalize(data);
        this._updateIfExistsElseAdd({ data, result });

        const current = this.countsList.getAttribute("data-filter");
        if (current !== "total" && current !== result) {
            updateCounts({ countsMap: this.countsMap, result: data._result, by: data.result || 1 });
            return;
        }

        this.tableMaker.addRow(data);
    }

    /**
     * Build a stable id for grouping rows across placeholder and events.
     * @param {Object} e
     * @returns {string}
     */
    _getId(e) {
        return this.tableMaker.getKeyFor(e);
    }

    _updateIfExistsElseAdd({data, result}) {
        const dataId = this._getId(data);
        const undefinedEntry = this._events.find(e => !e.remark && !e.result && this._getId(e) === dataId);
        const possibleRemark = this._possibleRemark(result);
        if (undefinedEntry) {
            undefinedEntry.remark = data.remark;
            undefinedEntry.result = data.result;
            undefinedEntry._result = data._result;
            undefinedEntry.__skipRowResult = false;
            undefinedEntry.__normalizedResult = result;
            undefinedEntry.___possibleRemark = possibleRemark;
            return;
        }

        this._events.push({ ...data, __normalizedResult: result, __skipCount: true, ___possibleRemark: possibleRemark });
    }

    _normalize(data) {
        if (this.tableMaker._isExcluded(data)) return "excluded";
        const v = (data._result || "notcovered").toLowerCase();
        return ["success", "error", "notcovered"].includes(v) ? /** @type {ResultType} */ (v) : "failed";
    }

    _possibleRemark(normalizedResult) {
        switch (normalizedResult) {
            case "excluded": return "Excluded";
            case "notcovered": return "Not Covered";
            default: return null
        }
    }

    /**
     * @param {Object[]} events
     */
    _updatePaths(events) {
        const groups = new Set(events.map(e => this.tableMaker.getMainGroupEntry(e)));
        for (const { value } of groups) {
            const coverage = this._coverage.get(value);
            if (coverage === undefined) continue;
            this.updateGroupCoverage({ group: value, coverage });
        }
    }

    /**

     * @param {string} group
     * @param {number} coverage
     */
    addCoverage(group, coverage) {
        this._coverage.set(group, coverage);
        try { this.updateGroupCoverage({ group, coverage }) } catch(e) { console.log(e) /* Can fail if path not on screen  */ }
    }

    addFinalCoverage(coverage) {
        const coverageHeader = this.tableMaker.target.querySelector("th[data-key=coverage")
        if (!coverageHeader) console.error(`Failed to find coverage table header to update final coverage of ${coverage}`);
        coverageHeader?.setAttribute("data-total", coverage);
    }

    setActuator(state) {
        const coverageHeader = this.tableMaker.target.querySelector("th[data-key=coverage")
        if (!coverageHeader) console.error(`Failed to find coverage table header to update actuator state to ${state}`);
        coverageHeader?.setAttribute("data-actuator", state === true ? "Enabled" : "Disabled");
    }

    getCurrentFilter() {
        const value = this.countsList.getAttribute("data-filter") || "total";
        return /** @type {ResultType} */ (value);
    }

    reset(tableMaker, selections) {
        const saved = this.countsList.getAttribute("data-filter");
        this.savedFilter = saved !== "total" ? /** @type {ResultType|null} */ (saved) : null;

        for (const el of this.countsList.querySelectorAll('[data-active]')) el.removeAttribute('data-active');
        this.countsList.setAttribute("data-filter", "total");
        this.countsList.querySelector('li[data-type="total"]')?.setAttribute('data-active', 'true');

        this.tableMaker = tableMaker;
        if (!selections || selections.length === 0) {
            this._events = [];
            this._coverage.clear();
            return;
        }

        const selectionsData = selections.map(selection => this.tableMaker._extractRowValues(selection));
        for (const data of selectionsData) this._resetEventsFor(data)
    }

    _resetEventsFor(data) {
        const dataId = this._getId(data);
        const entries = this._events.filter(e => this._getId(e) === dataId);
        for (const entry of entries) {
            entry.remark = undefined;
            entry.result = undefined;
            entry.__normalizedResult = "notcovered";
            entry.___possibleRemark = "Not Covered";
        }
    }

    onEnd(applyLastFilter) {
        const notCovered = this._events.filter(e => !e.result && !e.remark);
        for (const e of notCovered) Object.assign(e, { remark: e.___possibleRemark, result: 1, _result: e.__normalizedResult, __skipRowResult: false });
        if (this.savedFilter && applyLastFilter) this._applyFilter(this.savedFilter);
    }
}
