// biome-ignore lint/complexity/noStaticOnlyClass: <explanation>
class Workspace {
    static container = document.getElementById('screen-container');
    static sideBar = document.querySelector("aside");
    static template = document.getElementById('screen-template');
    static templateKafka = document.getElementById('screen-template-kafka');
    static templateConfig = document.getElementById('screen-template-config');
    static templateArazzo = document.getElementById('screen-template-arazzo')
    static templateSoap = document.getElementById('screen-template-soap');
    static PROXY_SCREEN_ID = 'proxy';
    static HOME_SCREEN_ID = 'default';
    static WORKFLOW_SCREEN_ID = 'workflow';
    static CONSTANT_SCREENS = new Set([this.HOME_SCREEN_ID, this.WORKFLOW_SCREEN_ID, this.PROXY_SCREEN_ID]);
    static SCREENS = new Set([this.PROXY_SCREEN_ID, this.HOME_SCREEN_ID]);
    static SCREEN_COUNT = 0;
    static _fileTreeEvents = ["ENTRY_CREATE", "ENTRY_DELETE"];

    /**
    * @param {string} screenId
    * @param {boolean} collapseSideBar=true
    */
    static async switchTo(screenId, collapseSideBar = true) {
        const targetScreen = document.getElementById(screenId);
        const activeScreen = document.querySelector('.screen[data-active]');
        const sideBar = document.querySelector('aside');

        if (!targetScreen || !activeScreen || targetScreen === activeScreen) return;
        if (!Workspace.CONSTANT_SCREENS.has(screenId)) {
            await Workspace.updateBackend(screenId, targetScreen, activeScreen);
        }

        const tIdx = +(targetScreen.dataset.index || 0);
        // @ts-ignore
        const aIdx = +activeScreen.dataset.index || 0;
        const inClass = tIdx < aIdx ? 'slide-in-top' : 'slide-in-bottom';
        const outClass = tIdx < aIdx ? 'slide-out-bottom' : 'slide-out-top';

        targetScreen.setAttribute('data-active', 'true');
        targetScreen.removeAttribute("inert");
        activeScreen.removeAttribute('data-active');
        activeScreen.setAttribute("inert", "true");

        targetScreen.classList.add(inClass);
        activeScreen.classList.add(outClass);

        targetScreen.addEventListener('animationend', (e) => {
            if (e.target === targetScreen) targetScreen.classList.remove(inClass);
        }, { once: true });

        activeScreen.addEventListener('animationend', (e) => {
            if (e.target === activeScreen) activeScreen.classList.remove(outClass);
        }, { once: true });

        if (collapseSideBar) sideBar?.setAttribute("aria-expanded", "false");
        document.dispatchEvent(new CustomEvent('screenChanged', { detail: { screenId } }));
    }

    static async updateBackend(screenId, targetScreen, activeScreen) {
        const fileType = targetScreen.getAttribute('data-file-type');
        const screenIdPath = targetScreen.getAttribute('id') || '';
        const isSpecmaticConfig = screenIdPath.endsWith('/specmatic.yaml') || screenIdPath.endsWith('\\specmatic.yaml') || screenIdPath === 'specmatic.yaml';

        if (isSpecmaticConfig) {
            try { initSpecEditor(targetScreen); } catch (e) { console.warn('Failed to init config editor', e); }
            return;
        }

        if (fileType === 'asyncapi' || fileType === "soap") {
            // Initialize Kafka or Soap mock table & controls when screen becomes active
            try { initMock(targetScreen); } catch (e) { console.warn('Failed to init async mock', e); }
            return;
        }

        if (fileType === "arazzo") {
            initSpecEditor(targetScreen);
            return;
        };

        const { error } = await makeHttpCall("/specifications/load", { method: "POST", body: { path: screenId, name: screenId } })
        if (error) {
            createAlert({ title: "Failed to load specification", message: error, type: "error" });
            return;
        }

        requestAnimationFrame(() => {
            const targetIframe = targetScreen.querySelector("iframe");
            targetIframe.src = targetIframe.getAttribute("data-src");
            const activeIframe = activeScreen.querySelector("iframe");
            if (activeIframe) activeIframe.src = 'about:blank';
        });
    }

    /**
     * @param {string} name
     * @param {string} path
     * @param {string} fileType
     * @param {string} displayablePath
     * @throws {Error}
     * @returns {HTMLDivElement}
     */
    static addScreen(name, path, fileType, displayablePath) {
        if (Workspace.SCREENS.has(path)) return false

        const index = String(Workspace.SCREEN_COUNT++);
        const screen = Workspace.screenUsingTemplate(name, path, index, fileType, displayablePath);
        if (!screen) return false;

        Workspace.SCREENS.add(path);
        Workspace.container?.appendChild(screen);
        return true;
    }

    static registerListeners() {
        Workspace.container?.addEventListener('click', async (e) => {
            const target = e.target;
            if (target?.tagName === 'LI' && target?.classList.contains('tab')) {
                const screen = target.closest('.screen');
                const targetClassName = target.getAttribute('data-type');
                Workspace.switchTab(screen, targetClassName);

                switch (targetClassName) {
                    case 'test':
                        Workspace.initTestScreen(screen);
                        break;
                    case 'mock':
                        initMock(screen);
                        break;
                    case 'spec':
                        initSpecEditor(screen);
                        break;
                    case 'example': {
                        registerListenerForMock(screen);
                        const fileType = screen.getAttribute('data-file-type');
                        if (fileType === 'arazzo') initInputsEditor(screen);
                        break;
                    }
                    default:
                        break;
                }

                e?.stopPropagation();
            }

            if (target?.tagName === 'BUTTON' && target?.classList.contains('run')) {
                const screen = target.closest('.screen');
                switch (target.dataset.type) {
                    case 'test':
                        Workspace.runTestScreen(screen);
                        break;
                    case 'mock':
                        await handleMock(screen);
                        break;
                    default:
                        break;
                }
                e?.stopPropagation();
            }
        });

        Workspace._watchServiceListener();
    }

    static initTestScreen(screen) {
        const specType = screen.getAttribute('data-file-type');
        switch (specType) {
            case 'openapi':
                initTest(screen);
                break;
            case 'arazzo':
                initWorkflowTest(screen);
                break;
            default:
                console.error(`Unknown spec type: ${specType}`);
                break;
        }
    }

    static runTestScreen(screen) {
        const specType = screen.getAttribute('data-file-type');
        switch (specType) {
            case 'openapi':
                runTest(screen);
                break;
            case 'arazzo':
                runWorkflowTest(screen);
                break;
            default:
                console.error(`Unknown spec type: ${specType}`);
                break;
        }
    }

    static reloadIframe(screen) {
        if (!screen.hasAttribute('data-active')) return;
        const screenIframe = screen.querySelector('iframe');
        if (!screenIframe) return;
        screenIframe.src = screenIframe.getAttribute('data-src');
    }

    static reloadMock(screen) {
        clearMockFor(screen)
    }

    static reloadTest(screen) {
        const specType = screen.getAttribute('data-file-type');
        switch (specType) {
            case 'openapi':
                clearTestFor(screen);
                break;
            case 'arazzo':
                clearWorkflowTestFor(screen);
                break;
            default:
                break;
        }
    }

    static screenUsingTemplate(name, path, index, fileType, displayablePath) {
        let screen;
        const isSpecmaticConfig = path.endsWith('/specmatic.yaml') || path.endsWith('\\specmatic.yaml') || path === 'specmatic.yaml';
        if (isSpecmaticConfig && Workspace.templateConfig) {
            screen = Workspace.templateConfig.content.cloneNode(true);
        } else {
            switch (fileType) {
                case 'asyncapi':
                    screen =  Workspace.templateKafka.content.cloneNode(true);
                    break;
                case 'arazzo':
                    if (!Workspace.templateArazzo) return;
                    screen = Workspace.templateArazzo.content.cloneNode(true);
                    break;
                case 'soap':
                    screen =  Workspace.templateSoap.content.cloneNode(true);
                    break;
                default:
                    screen = Workspace.template.content.cloneNode(true);
            }
        }

        const screenElement = screen.querySelector('.screen');
        screenElement.setAttribute('id', path);
        screenElement.setAttribute('data-index', index);
        screenElement.setAttribute('data-file-type', fileType);
        screen.querySelector('p[data-name]').textContent = name;
        screen.querySelector('span[data-path]').textContent = "File path: ./" + displayablePath;

        const iframe = screen.querySelector('iframe');

        if(iframe) {
            const examplesServerBase = iframe.getAttribute('data-examples-server-base');
            screen.querySelector('iframe').setAttribute('data-src', `${examplesServerBase}?specFile=${path}`);
        }

        try {
            if (typeof ensureBaseUrlInputState === 'function') {
                const baseUrlInput = screenElement.querySelector('#testBaseUrl');
                ensureBaseUrlInputState(baseUrlInput);
            }
        } catch (e) { /* ignore */ }

        try {
            if (typeof applyInitialConfigToScreen === 'function') {
                applyInitialConfigToScreen(screenElement);
            }
        } catch (e) { /* ignore */ }

        return screen;
    }

    static switchTab(screen, tabClassName) {
        const targetTab = screen.querySelector(`.tab[data-type=${tabClassName}]`);
        const activeTab = screen.querySelector('.tab[data-active]');
        const targetDiv = screen.querySelector(`div.${tabClassName}`)
        const activeDiv = screen.querySelector(`div.${activeTab.getAttribute("data-type")}`);

        if (targetTab === activeTab) return;
        screen.setAttribute('data-mode', targetTab.getAttribute("data-type"));

        activeTab.removeAttribute("data-active");
        activeDiv.setAttribute("inert", "true");

        targetTab.setAttribute('data-active', 'true');
        targetDiv.removeAttribute('inert');
    }

    static setScreenActive(screenId, type) {
        const targetScreen = document.getElementById(screenId);
        const target = targetScreen.querySelector(`.tab[data-type="${type}"]`);
        target.click();
    }

    static _watchServiceListener() {
        SseEventStreamer.subscribe({
            eventId: "WATCHER",
            callbacks: {
                 onData: (data) => {
                    if (Workspace._fileTreeEvents.includes(data.eventKind)) {
                        switch (data.eventKind) {
                            case "ENTRY_CREATE":
                                return addNode(data.node);
                            default:
                                return deleteNode(data.node.path);
                        }
                    }
                    Workspace.debouncedReload(data);
                }
            }
        })

        SseEventStreamer.subscribe({
            eventId: "WATCHER_RESET",
            callbacks: {
                onData: () => {
                    try {
                        Workspace.switchTo(Workspace.HOME_SCREEN_ID, true);

                        // Clear process/activity UI (cards, badge) if utility is available
                        if (window.resetProcessUI) {
                            try { window.resetProcessUI(); } catch (e) { console.warn('resetProcessUI failed', e); }
                        }

                        const toRemove = [...Workspace.SCREENS].filter(id => !Workspace.CONSTANT_SCREENS.has(id));
                        toRemove.forEach(id => {
                            const elem = document.getElementById(id);
                            if (elem) elem.remove();
                            Workspace.SCREENS.delete(id);
                        });

                        Workspace.SCREEN_COUNT = Workspace.CONSTANT_SCREENS.size;
                        if (window.initializeSpecTree) {
                            window.initializeSpecTree();
                        }
                    } catch (e) {
                        console.error("Failed handling WATCHER_RESET", e);
                    }
                }
            }
        })
    }

    static debouncedReload = leadingKeyedDebounce((data) => {
        const screen = document.getElementById(data.node.path);
        if (!screen) return;
        Workspace.reloadIframe(screen);
        Workspace.reloadMock(screen)
        Workspace.reloadTest(screen);
    }, 250, (data) => data.node.path);

    static updateInfoBox(screen, text, query) {
       if (!screen) return;

       // Find the .mock div inside .details in the given screen
       const div = screen.querySelector(query);
       if (!div) return;

       // Specifically target the .info div inside that .mock div
       const infoDiv = div.querySelector(".info-message-box");
       if (!infoDiv) return;

       // Clear previous content
       infoDiv.innerHTML = "";

       if (text && text.trim().length > 0) {
           infoDiv.textContent = text;
           infoDiv.style.opacity = 1; // ensure visible
       } else {
           // Hide the info div if no message
           infoDiv.style.opacity = 0;
       }
    }

}
